/*
 * Copyright (C) 2000-2026 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.testbench.elements;

import java.util.List;
import java.util.stream.Collectors;

import org.openqa.selenium.WebElement;

import com.vaadin.testbench.By;
import com.vaadin.testbench.elementsbase.ServerClass;

/**
 * TestBench element for Vaadin Charts.
 *
 * @since 8.30
 */
@ServerClass("com.vaadin.addon.charts.Chart")
public class ChartElement extends AbstractComponentElement {

    /**
     * Gets the SVG element inside the chart.
     *
     * @return the SVG WebElement
     */
    public WebElement getSvg() {
        return findElement(By.tagName("svg"));
    }

    /**
     * Gets the menu button element inside the chart.
     *
     * @return the menu button WebElement, a SVG element
     */
    public WebElement getMenuButton() {
        return findElement(By.className("highcharts-button"));
    }

    /**
     * Gets the context menu overlay element inside the chart.
     *
     * @return the menu overlay WebElement, a DIV element
     */
    public WebElement getMenuOverlay() {
        return findElement(By.className("highcharts-contextmenu"));
    }

    /**
     * Gets the menu item elements inside the chart menu overlay.
     *
     * @return the menu item WebElements
     */
    public List<WebElement> getMenuItems() {
        WebElement menu = getMenuOverlay().findElement(By.tagName("div"));
        return menu.findElements(By.tagName("div"));
    }

    /**
     * Gets the legend item elements inside the chart.
     *
     * @return the legend item WebElements, SVG elements
     */
    public List<WebElement> getLegendItems() {
        return findElements(By.className("highcharts-legend-item"));
    }

    /**
     * Gets the series elements inside the chart.
     *
     * @return the series WebElements, SVG elements
     */
    public List<WebElement> getSeries() {
        return findElements(By.className("highcharts-series"));
    }

    /**
     * Gets the Y axis title texts inside the chart.
     *
     * @return the list of Y axis title texts
     */
    public List<String> getYAxisTitles() {
        return findElements(By.className("highcharts-yaxis-title")).stream()
                .map(WebElement::getText).collect(Collectors.toList());
    }

    /**
     * Gets the Y axis label texts inside the chart.
     *
     * @param index
     *            the Y axis index, 0 = first
     * @return the list of Y axis label texts in the ascending order
     */
    public List<String> getYAxisLabels(int index) {
        List<WebElement> yAxes = findElements(
                By.className("highcharts-yaxis-labels"));
        WebElement labelsContainer = yAxes.get(index);
        List<WebElement> texts = labelsContainer
                .findElements(By.tagName("text"));
        return texts.stream().map(ChartElement::getLabelText)
                .collect(Collectors.toList());
    }

    /**
     * Gets the X axis label texts inside the chart.
     *
     * @return the list of X axis label texts
     */
    public List<String> getXAxisLabels() {
        WebElement labelsContainer = findElement(
                By.className("highcharts-xaxis-labels"));
        List<WebElement> texts = labelsContainer
                .findElements(By.tagName("text"));
        return texts.stream().map(ChartElement::getLabelText)
                .collect(Collectors.toList());
    }

    /**
     * Gets the text content of a label element (which may contain multiple
     * tspan elements).
     *
     * @param text
     *            the label WebElement
     * @return the combined text content of the label
     */
    protected static String getLabelText(WebElement text) {
        return text.findElements(By.tagName("tspan")).stream()
                .map(WebElement::getText).collect(Collectors.joining(" "));
    }

    /**
     * Gets the data label texts inside the chart.
     *
     * @return the list of data label texts
     */
    public List<String> getDataLabels() {
        return findElements(By.className("highcharts-data-labels")).stream()
                .flatMap(dl -> dl.findElements(By.tagName("tspan")).stream())
                .map(WebElement::getText).collect(Collectors.toList());
    }

    /**
     * Gets the chart title text.
     *
     * @return the chart title text
     */
    public String getTitle() {
        WebElement title = findElement(By.className("highcharts-title"));
        return getLabelText(title);
    }

    /**
     * Gets the chart subtitle text.
     *
     * @return the chart subtitle text
     */
    public String getSubTitle() {
        WebElement subtitle = findElement(By.className("highcharts-subtitle"));
        return getLabelText(subtitle);
    }
}
