/*
 * Copyright (C) 2000-2025 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.testbench.elements;

import java.util.ArrayList;
import java.util.List;

import org.openqa.selenium.By;
import org.openqa.selenium.WebElement;
import org.openqa.selenium.support.ui.Select;

import com.vaadin.testbench.elementsbase.ServerClass;

@ServerClass("com.vaadin.ui.ListSelect")
public class ListSelectElement extends AbstractSelectElement {

    private Select select;
    private static By bySelect = By.tagName("select");
    private WebElement selectElement;

    @Override
    protected void init() {
        super.init();
        selectElement = findElement(bySelect);
        select = new Select(selectElement);
    }

    /**
     * Selects the option with the given text.
     * <p>
     * For a ListSelect in multi select mode, adds the given option(s) to the
     * current selection.
     *
     * @param text
     *            the text of the option
     */
    public void selectByText(String text) {
        if (isReadOnly()) {
            throw new ReadOnlyException();
        }

        select.selectByVisibleText(text);
    }

    /**
     * Deselects the option(s) with the given text.
     *
     * @param text
     *            the text of the option
     */
    public void deselectByText(String text) {
        if (isReadOnly()) {
            throw new ReadOnlyException();
        }
        select.deselectByVisibleText(text);
    }

    /**
     * Gets a list of the texts shown for all options.
     *
     * @return a list of option texts
     */
    public List<String> getOptions() {
        List<String> options = new ArrayList<String>();
        for (WebElement webElement : select.getOptions()) {
            options.add(webElement.getText());
        }
        return options;
    }

    /**
     * Clear operation is not supported for List Select. This operation has no
     * effect on List Select element.
     */
    @Override
    public void clear() {
        super.clear();
    }

    /**
     * Return value of the list select element.
     *
     * @return value of the list select element
     */
    public String getValue() {
        return select.getFirstSelectedOption().getText();
    }

    /**
     * Gets the {@code <select>} element inside the component.
     *
     * @return the select element inside the component
     * @since 8.1.1
     */
    public WebElement getSelectElement() {
        return selectElement;
    }

}
