/*
 * Copyright 2000-2025 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.flow.spring.security;

import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletRequestWrapper;

import java.security.Principal;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.web.servlet.ServletRegistrationBean;
import org.springframework.http.HttpMethod;
import org.springframework.security.web.servlet.util.matcher.PathPatternRequestMatcher;
import org.springframework.security.web.util.matcher.RequestMatcher;
import org.springframework.stereotype.Component;

import com.vaadin.flow.internal.hilla.EndpointRequestUtil;
import com.vaadin.flow.internal.hilla.FileRouterRequestUtil;
import com.vaadin.flow.router.Location;
import com.vaadin.flow.router.QueryParameters;
import com.vaadin.flow.router.Router;
import com.vaadin.flow.router.internal.NavigationRouteTarget;
import com.vaadin.flow.router.internal.RouteTarget;
import com.vaadin.flow.server.HandlerHelper;
import com.vaadin.flow.server.RouteRegistry;
import com.vaadin.flow.server.VaadinService;
import com.vaadin.flow.server.VaadinServletService;
import com.vaadin.flow.server.auth.AccessCheckDecision;
import com.vaadin.flow.server.auth.AccessCheckResult;
import com.vaadin.flow.server.auth.AnonymousAllowed;
import com.vaadin.flow.server.auth.NavigationAccessControl;
import com.vaadin.flow.server.auth.NavigationContext;
import com.vaadin.flow.spring.AuthenticationUtil;
import com.vaadin.flow.spring.SpringServlet;
import com.vaadin.flow.spring.VaadinConfigurationProperties;

/**
 * Contains utility methods related to request handling.
 */
@Component
public class RequestUtil {

    private static final ThreadLocal<Boolean> ROUTE_PATH_MATCHER_RUNNING = new ThreadLocal<>();

    @Autowired
    private ObjectProvider<NavigationAccessControl> accessControl;

    @Autowired
    private VaadinConfigurationProperties configurationProperties;

    @Autowired(required = false)
    private EndpointRequestUtil endpointRequestUtil;

    @Autowired(required = false)
    private FileRouterRequestUtil fileRouterRequestUtil;

    @Autowired
    private ServletRegistrationBean<SpringServlet> springServletRegistration;

    private WebIconsRequestMatcher webIconsRequestMatcher;

    /**
     * Checks whether the request is an internal request.
     *
     * An internal request is one that is needed for all Vaadin applications to
     * function, e.g. UIDL or init requests.
     *
     * Note that bootstrap requests for any route or static resource requests
     * are not internal, neither are resource requests for the JS bundle.
     *
     * @param request
     *            the servlet request
     * @return {@code true} if the request is Vaadin internal, {@code false}
     *         otherwise
     */
    public boolean isFrameworkInternalRequest(HttpServletRequest request) {
        String vaadinMapping = configurationProperties.getUrlMapping();
        return HandlerHelper.isFrameworkInternalRequest(vaadinMapping, request);
    }

    /**
     * Checks whether the request targets an endpoint.
     *
     * @param request
     *            the servlet request
     * @return {@code true} if the request is targeting an enpoint,
     *         {@code false} otherwise
     */
    public boolean isEndpointRequest(HttpServletRequest request) {
        if (endpointRequestUtil != null) {
            return endpointRequestUtil.isEndpointRequest(request);
        }
        return false;
    }

    /**
     * Checks whether the request targets an endpoint that is public, i.e.
     * marked as @{@link AnonymousAllowed}.
     *
     * @param request
     *            the servlet request
     * @return {@code true} if the request is targeting an anonymous endpoint,
     *         {@code false} otherwise
     */
    public boolean isAnonymousEndpoint(HttpServletRequest request) {
        if (endpointRequestUtil != null) {
            return endpointRequestUtil.isAnonymousEndpoint(request);
        }
        return false;
    }

    /**
     * Checks if the request targets a Hilla view that is allowed according to
     * its configuration and the current user.
     *
     * @param request
     *            the HTTP request to check
     * @return {@code true} if the request corresponds to an accessible Hilla
     *         view, {@code false} otherwise
     */
    public boolean isAllowedHillaView(HttpServletRequest request) {
        if (fileRouterRequestUtil != null) {
            return fileRouterRequestUtil.isRouteAllowed(request);
        }
        return false;
    }

    /**
     * Checks whether the request targets a Flow route that is public, i.e.
     * marked as @{@link AnonymousAllowed}.
     *
     * @param request
     *            the servlet request
     * @return {@code true} if the request is targeting an anonymous route,
     *         {@code false} otherwise
     */
    public boolean isAnonymousRoute(HttpServletRequest request) {
        if (ROUTE_PATH_MATCHER_RUNNING.get() == null) {
            ROUTE_PATH_MATCHER_RUNNING.set(Boolean.TRUE);
            try {
                return isAnonymousRouteInternal(
                        PrincipalAwareRequestWrapper.wrap(request));
            } finally {
                ROUTE_PATH_MATCHER_RUNNING.remove();
            }
        }
        // A route path check is already in progress for the current request
        // this matcher should be considered only once, since for alias check
        // we are interested only in the other matchers
        return false;
    }

    /**
     * Checks whether the request targets a Flow route secured with navigation
     * access control.
     *
     * @param request
     *            the servlet request
     * @return {@code true} if the request is targeting a Flow route secured
     *         with navigation access control, {@code false} otherwise
     */
    public boolean isSecuredFlowRoute(HttpServletRequest request) {
        return isSecuredFlowRouteInternal(request);
    }

    /**
     * Checks whether the request targets a custom PWA icon or Favicon path.
     *
     * @param request
     *            the servlet request
     * @return {@code true} if the request is targeting a custom PWA icon or a
     *         custom favicon path, {@code false} otherwise
     */
    public boolean isCustomWebIcon(HttpServletRequest request) {
        if (webIconsRequestMatcher == null) {
            VaadinServletService vaadinService = springServletRegistration
                    .getServlet().getService();
            if (vaadinService != null) {
                webIconsRequestMatcher = new WebIconsRequestMatcher(
                        vaadinService, configurationProperties.getUrlMapping());
            } else {
                getLogger().debug(
                        "WebIconsRequestMatcher cannot be created because VaadinService is not yet available. "
                                + "This may happen after a hot-reload, and can cause requests for icons to be blocked by Spring Security.");
                return false;
            }
        }
        return webIconsRequestMatcher.matches(request);
    }

    /**
     * Utility to create {@link RequestMatcher}s from ant patterns.
     * <p>
     * Since org.springframework.security.web.util.matcher.AntPathRequestMatcher
     * is deprecated and will be removed, callers of this method should be
     * updated to use {@link PathPatternRequestMatcher} instead.
     *
     * <pre>
     * {@code
     *  var matcherBuilder = PathPatternRequestMatcher.withDefaults():
     *  var requestMatcher = matcherBuilder.match(path);
     * }
     * </pre>
     *
     * @param patterns
     *            and patterns
     * @return an array or {@link RequestMatcher} instances for the given
     *         patterns.
     * @deprecated {@code AntPathRequestMatcher} is deprecated and marked for
     *             removal. This method is deprecated without direct
     *             replacement; use {@code PathPatternRequestMatcher} instead.
     */
    @Deprecated(since = "24.8", forRemoval = true)
    public static RequestMatcher[] antMatchers(String... patterns) {
        return Stream.of(patterns).map(PathPatternRequestMatcher::pathPattern)
                .toArray(RequestMatcher[]::new);
    }

    /**
     * Utility to create {@link RequestMatcher}s for a Vaadin routes, using ant
     * patterns and HTTP get method.
     * <p>
     * Since org.springframework.security.web.util.matcher.AntPathRequestMatcher
     * is deprecated and will be removed, callers of this method should be
     * updated to use {@link PathPatternRequestMatcher} instead.
     *
     * <pre>
     * {@code
     *  var matcherBuilder = PathPatternRequestMatcher.withDefaults():
     *  var requestMatcher = matcherBuilder.match(HttpMethod.GET, path);
     * }
     * </pre>
     *
     * @param patterns
     *            ANT patterns
     * @return an array or {@link RequestMatcher} instances for the given
     *         patterns.
     * @deprecated {@code AntPathRequestMatcher} is deprecated and marked for
     *             removal. This method is deprecated without direct
     *             replacement; use {@code PathPatternRequestMatcher} instead.
     */
    @Deprecated(since = "24.8", forRemoval = true)
    public static RequestMatcher[] routeMatchers(String... patterns) {
        return Stream.of(patterns).map(
                p -> PathPatternRequestMatcher.pathPattern(HttpMethod.GET, p))
                .toArray(RequestMatcher[]::new);
    }

    /**
     * Wraps a given {@link RequestMatcher} to ensure requests are processed
     * with the principal awareness provided by
     * {@link RequestUtil.PrincipalAwareRequestWrapper}.
     *
     * @param matcher
     *            the {@link RequestMatcher} to be wrapped
     * @return a {@link RequestMatcher} that processes requests using a
     *         {@link RequestUtil.PrincipalAwareRequestWrapper} for principal
     *         awareness
     */
    public static RequestMatcher principalAwareRequestMatcher(
            RequestMatcher matcher) {
        return request -> matcher.matches(
                RequestUtil.PrincipalAwareRequestWrapper.wrap(request));
    }

    private boolean isSecuredFlowRouteInternal(HttpServletRequest request) {
        NavigationAccessControl navigationAccessControl = accessControl
                .getObject();
        if (!navigationAccessControl.isEnabled()) {
            return false;
        }
        return isFlowRouteInternal(request);
    }

    private boolean isFlowRouteInternal(HttpServletRequest request) {
        String path = getRequestRoutePath(request);
        if (path == null)
            return false;

        SpringServlet servlet = springServletRegistration.getServlet();
        VaadinService service = servlet.getService();
        if (service == null) {
            // The service has not yet been initialized. We cannot know if this
            // is an authenticated route, so better say it is not.
            return false;
        }
        Router router = service.getRouter();
        RouteRegistry routeRegistry = router.getRegistry();

        NavigationRouteTarget target = routeRegistry
                .getNavigationRouteTarget(path);
        if (target == null) {
            return false;
        }
        RouteTarget routeTarget = target.getRouteTarget();
        if (routeTarget == null) {
            return false;
        }
        Class<? extends com.vaadin.flow.component.Component> targetView = routeTarget
                .getTarget();
        return targetView != null;
    }

    private boolean isAnonymousRouteInternal(HttpServletRequest request) {
        String path = getRequestRoutePath(request);
        if (path == null)
            return false;

        SpringServlet servlet = springServletRegistration.getServlet();
        VaadinService service = servlet.getService();
        if (service == null) {
            // The service has not yet been initialized. We cannot know if this
            // is an anonymous route, so better say it is not.
            return false;
        }
        Router router = service.getRouter();
        RouteRegistry routeRegistry = router.getRegistry();

        NavigationRouteTarget target = routeRegistry
                .getNavigationRouteTarget(path);
        if (target == null) {
            return false;
        }
        RouteTarget routeTarget = target.getRouteTarget();
        if (routeTarget == null) {
            return false;
        }
        Class<? extends com.vaadin.flow.component.Component> targetView = routeTarget
                .getTarget();
        if (targetView == null) {
            return false;
        }

        boolean productionMode = service.getDeploymentConfiguration()
                .isProductionMode();
        NavigationAccessControl navigationAccessControl = accessControl
                .getObject();
        if (!navigationAccessControl.isEnabled()) {
            String message = "Navigation Access Control is disabled. Cannot determine if {} refers to a public view, thus access is denied. Please add an explicit request matcher rule for this URL.";
            if (productionMode) {
                getLogger().debug(message, path);
            } else {
                getLogger().info(message, path);
            }
            return false;
        }

        NavigationContext navigationContext = new NavigationContext(router,
                targetView,
                new Location(path,
                        QueryParameters.full(request.getParameterMap())),
                target.getRouteParameters(), null, role -> false, false);

        AccessCheckResult result = navigationAccessControl
                .checkAccess(navigationContext, productionMode);
        boolean isAllowed = result.decision() == AccessCheckDecision.ALLOW;
        if (isAllowed) {
            getLogger().debug("{} refers to a public view", path);
        } else {
            getLogger().debug(
                    "Access to {} denied by Flow navigation access control. {}",
                    path, result.reason());
        }
        return isAllowed;
    }

    private String getRequestRoutePath(HttpServletRequest request) {
        String vaadinMapping = configurationProperties.getUrlMapping();
        String requestedPath = HandlerHelper
                .getRequestPathInsideContext(request);
        return HandlerHelper
                .getPathIfInsideServlet(vaadinMapping, requestedPath)
                // Requested path includes a beginning "/" but route mapping is
                // done without one
                .map(path -> path.startsWith("/") ? path.substring(1) : path)
                .orElse(null);
    }

    /**
     * Gets the url mapping for the Vaadin servlet.
     *
     * @return the url mapping
     * @see VaadinConfigurationProperties#getUrlMapping()
     */
    public String getUrlMapping() {
        return configurationProperties.getUrlMapping();
    }

    /**
     * Prepends to the given {@code path} with the configured url mapping.
     *
     * A {@literal null} path is treated as empty string; the same applies for
     * url mapping.
     *
     * @return the path with prepended url mapping.
     * @see VaadinConfigurationProperties#getUrlMapping()
     */
    public String applyUrlMapping(String path) {
        return applyUrlMapping(configurationProperties.getUrlMapping(), path);
    }

    /**
     * Prepends to the given {@code path} with the servlet path prefix from
     * input url mapping.
     *
     * A {@literal null} path is treated as empty string; the same applies for
     * url mapping.
     *
     * @return the path with prepended url mapping.
     * @see VaadinConfigurationProperties#getUrlMapping()
     */
    static String applyUrlMapping(String urlMapping, String path) {
        if (urlMapping == null) {
            urlMapping = "";
        } else {
            // remove potential / or /* at the end of the mapping
            urlMapping = urlMapping.replaceFirst("/\\*?$", "");
        }
        if (path == null) {
            path = "";
        } else if (path.startsWith("/")) {
            path = path.substring(1);
        }
        return urlMapping + "/" + path;
    }

    /**
     * A wrapper for {@link HttpServletRequest} that provides additional
     * functionality to handle the user principal retrieval in a safer manner.
     * <p>
     * This class extends {@link HttpServletRequestWrapper} and overrides its
     * {@code getUserPrincipal()} method to handle cases where the operation
     * might not be supported by the underlying {@link HttpServletRequest}, for
     * example when called by a Spring request matcher in the context of
     * {@code WebInvocationPrivilegeEvaluator} permissions evaluation.
     */
    static class PrincipalAwareRequestWrapper
            extends HttpServletRequestWrapper {

        private PrincipalAwareRequestWrapper(HttpServletRequest request) {
            super(request);
        }

        @Override
        public Principal getUserPrincipal() {
            try {
                return super.getUserPrincipal();
            } catch (UnsupportedOperationException e) {
                return AuthenticationUtil.getSecurityHolderAuthentication();
            }
        }

        static HttpServletRequest wrap(HttpServletRequest request) {
            if (request instanceof PrincipalAwareRequestWrapper) {
                return request;
            }
            HttpServletRequest maybeWrapper = request;
            while (maybeWrapper instanceof HttpServletRequestWrapper wrapper) {
                if (wrapper instanceof PrincipalAwareRequestWrapper) {
                    return request;
                }
                maybeWrapper = (HttpServletRequest) wrapper.getRequest();
            }
            return new PrincipalAwareRequestWrapper(request);
        }
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }

}
