/*
 * Copyright (C) 2000-2026 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.shared.ui.richtextarea;

import java.io.Serializable;

/**
 * Supporting data structure for snippets feature.
 * 
 * @since 8.16
 */
public class Snippet implements Serializable {

    private static final int MAX_TITLE_LENGTH = 20;

    public String title;
    public String text;

    /**
     * Creates an empty snippet
     */
    public Snippet() {
    	this(null);
    }

    /**
     * Creates a snippet with a specified body text. Title is automatically
     * generated from the first 20 characters in the text, with ellipsis added
     * if the length exceeds 20 characters.
     * 
     * @param text
     *            body text, may contain HTML markup
     */
    public Snippet(String text) {
    	this(null,text);
    }

    /**
     * Creates a snippet with a specified title and text. The title can be of
     * arbitrary length. The body text may contain HTML markup.
     * 
     * @param title
     *            title of the Snippet. May be of arbitrary length.
     * @param text
     *            body text, may contain HTML markup
     */
    public Snippet(String title, String text) {
        this.text = text != null ? text : "";
        this.title = title != null ? title : formatTextAsTitle(this.text);
    }

    /**
     * Get the set title text (either a user-specified string or up to 20
     * characters of the main snippet text.
     * 
     * @return a string
     */
    public String getTitle() {
        return title;
    }

    /**
     * Get the snippet content text.
     * 
     * @return a string
     */
    public String getText() {
        return text;
    }

    private static String formatTextAsTitle(String text) {
        return text.length() > MAX_TITLE_LENGTH
                ? text.substring(0, MAX_TITLE_LENGTH - 3) + "..."
                : text;
    }
}
