/*
 * Copyright (C) 2000-2026 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */

package com.vaadin.event;

import java.io.Serializable;
import java.lang.reflect.Method;

import com.vaadin.shared.Registration;

/**
 * <p>
 * Interface for classes supporting registration of methods as event receivers.
 * </p>
 *
 * <p>
 * For more information on the inheritable event mechanism see the
 * {@link com.vaadin.event com.vaadin.event package documentation}.
 * </p>
 *
 * @author Vaadin Ltd.
 * @since 3.0
 */
public interface MethodEventSource extends Serializable {

    /**
     * Registers a new event listener with the specified activation method to
     * listen events generated by this component. If the activation method does
     * not have any arguments the event object will not be passed to it when
     * it's called.
     *
     * <p>
     * For more information on the inheritable event mechanism see the
     * {@link com.vaadin.event com.vaadin.event package documentation}.
     * </p>
     *
     * @param eventType
     *            the type of the listened event. Events of this type or its
     *            subclasses activate the listener.
     * @param object
     *            the object instance who owns the activation method.
     * @param method
     *            the activation method.
     * @return a registration object for removing the listener
     * @throws IllegalArgumentException
     *             unless <code>method</code> has exactly one match in
     *             <code>object</code>
     * @throws NullPointerException
     *             if {@code object} is {@code null}
     * @since 8.0
     */
    @Deprecated
    public Registration addListener(Class<?> eventType, Object object,
            Method method);

    /**
     * Registers a new event listener with the specified activation method to
     * listen events generated by this component. If the activation method does
     * not have any arguments the event object will not be passed to it when
     * it's called.
     *
     * <p>
     * For more information on the inheritable event mechanism see the
     * {@link com.vaadin.event com.vaadin.event package documentation}.
     * </p>
     *
     * @param eventType
     *            the type of the listened event. Events of this type or its
     *            subclasses activate the listener.
     * @param listener
     *            the listener instance who owns the activation method.
     * @param method
     *            the activation method.
     * @return a registration object for removing the listener
     * @throws IllegalArgumentException
     *             unless <code>method</code> has exactly one match in
     *             <code>object</code>
     * @throws NullPointerException
     *             if {@code object} is {@code null}
     * @since 8.12
     */
    public Registration addListener(Class<?> eventType,
            SerializableEventListener listener, Method method);

    /**
     * Registers a new listener with the specified activation method to listen
     * events generated by this component. If the activation method does not
     * have any arguments the event object will not be passed to it when it's
     * called.
     *
     * <p>
     * This version of <code>addListener</code> gets the name of the activation
     * method as a parameter. The actual method is reflected from
     * <code>object</code>, and unless exactly one match is found,
     * <code>java.lang.IllegalArgumentException</code> is thrown.
     * </p>
     *
     * <p>
     * For more information on the inheritable event mechanism see the
     * {@link com.vaadin.event com.vaadin.event package documentation}.
     * </p>
     *
     * @param eventType
     *            the type of the listened event. Events of this type or its
     *            subclasses activate the listener.
     * @param object
     *            the object instance who owns the activation method.
     * @param methodName
     *            the name of the activation method.
     * @return a registration object for removing the listener
     * @throws IllegalArgumentException
     *             unless <code>method</code> has exactly one match in
     *             <code>object</code>
     * @throws NullPointerException
     *             if {@code object} is {@code null}
     * @since 8.0
     */
    @Deprecated
    public Registration addListener(Class<?> eventType, Object object,
            String methodName);

    /**
     * Registers a new listener with the specified activation method to listen
     * events generated by this component. If the activation method does not
     * have any arguments the event object will not be passed to it when it's
     * called.
     *
     * <p>
     * This version of <code>addListener</code> gets the name of the activation
     * method as a parameter. The actual method is reflected from
     * <code>listener</code>, and unless exactly one match is found,
     * <code>java.lang.IllegalArgumentException</code> is thrown.
     * </p>
     *
     * <p>
     * For more information on the inheritable event mechanism see the
     * {@link com.vaadin.event com.vaadin.event package documentation}.
     * </p>
     *
     * @param eventType
     *            the type of the listened event. Events of this type or its
     *            subclasses activate the listener.
     * @param listener
     *            the listener instance who owns the activation method.
     * @param methodName
     *            the name of the activation method.
     * @return a registration object for removing the listener
     * @throws IllegalArgumentException
     *             unless <code>method</code> has exactly one match in
     *             <code>object</code>
     * @throws NullPointerException
     *             if {@code object} is {@code null}
     * @since 8.12
     */
    public Registration addListener(Class<?> eventType,
            SerializableEventListener listener, String methodName);

    /**
     * Removes all registered listeners matching the given parameters. Since
     * this method receives the event type and the listener object as
     * parameters, it will unregister all <code>object</code>'s methods that are
     * registered to listen to events of type <code>eventType</code> generated
     * by this component.
     *
     * <p>
     * For more information on the inheritable event mechanism see the
     * {@link com.vaadin.event com.vaadin.event package documentation}.
     * </p>
     *
     * @param eventType
     *            the exact event type the <code>object</code> listens to.
     * @param target
     *            the target object that has registered to listen to events of
     *            type <code>eventType</code> with one or more methods.
     */
    @Deprecated
    public void removeListener(Class<?> eventType, Object target);

    /**
     * Removes all registered listeners matching the given parameters. Since
     * this method receives the event type and the listener object as
     * parameters, it will unregister all <code>object</code>'s methods that are
     * registered to listen to events of type <code>eventType</code> generated
     * by this component.
     *
     * <p>
     * For more information on the inheritable event mechanism see the
     * {@link com.vaadin.event com.vaadin.event package documentation}.
     * </p>
     *
     * @param eventType
     *            the exact event type the <code>object</code> listens to.
     * @param listener
     *            the listener that has registered to listen to events of type
     *            <code>eventType</code> with one or more methods.
     * @since 8.12
     */
    public void removeListener(Class<?> eventType,
            SerializableEventListener listener);

    /**
     * Removes one registered listener method. The given method owned by the
     * given object will no longer be called when the specified events are
     * generated by this component.
     *
     * <p>
     * For more information on the inheritable event mechanism see the
     * {@link com.vaadin.event com.vaadin.event package documentation}.
     * </p>
     *
     * @param eventType
     *            the exact event type the <code>object</code> listens to.
     * @param target
     *            the target object that has registered to listen to events of
     *            type eventType with one or more methods.
     * @param method
     *            the method owned by the target that's registered to listen to
     *            events of type eventType.
     * @deprecated use a {@link Registration} returned by
     *             {@link #addListener(Class, Object, Method)}
     */
    @Deprecated
    public void removeListener(Class<?> eventType, Object target,
            Method method);

    /**
     * Removes one registered listener method. The given method owned by the
     * given object will no longer be called when the specified events are
     * generated by this component.
     *
     * <p>
     * This version of <code>removeListener</code> gets the name of the
     * activation method as a parameter. The actual method is reflected from the
     * target, and unless exactly one match is found,
     * <code>java.lang.IllegalArgumentException</code> is thrown.
     * </p>
     *
     * <p>
     * For more information on the inheritable event mechanism see the
     * {@link com.vaadin.event com.vaadin.event package documentation}.
     * </p>
     *
     * @param eventType
     *            the exact event type the <code>object</code> listens to.
     * @param target
     *            the target object that has registered to listen to events of
     *            type <code>eventType</code> with one or more methods.
     * @param methodName
     *            the name of the method owned by <code>target</code> that's
     *            registered to listen to events of type <code>eventType</code>.
     * @deprecated use a {@link Registration} returned by
     *             {@link #addListener(Class, Object, String)}
     */
    @Deprecated
    public void removeListener(Class<?> eventType, Object target,
            String methodName);
}
