/*
 * Copyright (C) 2000-2026 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.data.provider;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import com.vaadin.shared.data.sort.SortDirection;

/**
 * Helper class for constructing SortOrders.
 *
 * @author Vaadin Ltd
 * @since 8.0
 */
public abstract class Sort implements Serializable {

    /**
     * SortBuilder is a helper class with fluent API for constructing sort order
     * lists. When the sort order is ready to be passed on, calling
     * {@link #build()} will create the list of sort orders
     *
     *
     * @see Sort
     * @see Sort#asc(String)
     * @see Sort#desc(String)
     * @see #build()
     */
    public static class SortBuilder implements Serializable {
        private List<QuerySortOrder> sortOrder = new ArrayList<>();

        /**
         * Constructs an empty SortBuilder.
         */
        protected SortBuilder() {
        }

        /**
         * Appends sorting with ascending sort direction.
         *
         * @param by
         *            the object to sort by
         * @return this sort builder
         */
        public SortBuilder thenAsc(String by) {
            return append(by, SortDirection.ASCENDING);
        }

        /**
         * Appends sorting with descending sort direction.
         *
         * @param by
         *            the object to sort by
         * @return this sort builder
         */
        public SortBuilder thenDesc(String by) {
            return append(by, SortDirection.DESCENDING);
        }

        /**
         * Appends sorting with given sort direction.
         *
         * @param by
         *            the object to sort by
         * @param direction
         *            the sort direction
         *
         * @return this sort builder
         */
        protected SortBuilder append(String by, SortDirection direction) {
            sortOrder.add(new QuerySortOrder(by, direction));
            return this;
        }

        /**
         * Returns an unmodifiable list of the current sort order in this sort
         * builder.
         *
         * @return the unmodifiable sort order list
         */
        public List<QuerySortOrder> build() {
            return Collections.unmodifiableList(sortOrder);
        }
    }

    /**
     * Creates a new sort builder with given sorting using ascending sort
     * direction.
     *
     * @param by
     *            the object to sort by
     *
     * @return the sort builder
     */
    public static SortBuilder asc(String by) {
        return new SortBuilder().thenAsc(by);
    }

    /**
     * Creates a new sort builder with given sorting using descending sort
     * direction.
     *
     * @param by
     *            the object to sort by
     *
     * @return the sort builder
     */
    public static SortBuilder desc(String by) {
        return new SortBuilder().thenDesc(by);
    }
}
