/*
 * Copyright (C) 2000-2026 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.event;

import java.io.Serializable;
import java.util.List;

import com.vaadin.data.provider.DataProvider;
import com.vaadin.data.provider.SortOrder;
import com.vaadin.shared.Registration;
import com.vaadin.ui.Component;

/**
 * Event describing a change in sorting of a {@link DataProvider}. Fired by
 * {@link SortNotifier SortNotifiers}.
 *
 * @see SortListener
 * @see SortOrder
 * @param <T>
 *            the type of the sorting information
 *
 * @since 8.0
 * @author Vaadin Ltd
 */
public class SortEvent<T extends SortOrder<?>> extends Component.Event
        implements HasUserOriginated {

    private final List<T> sortOrder;
    private final boolean userOriginated;

    /**
     * Creates a new sort order change event with a sort order list.
     *
     * @param source
     *            the component from which the event originates
     * @param sortOrder
     *            the new sort order list
     * @param userOriginated
     *            <code>true</code> if event is a result of user interaction,
     *            <code>false</code> if from API call
     */
    public SortEvent(Component source, List<T> sortOrder,
            boolean userOriginated) {
        super(source);
        this.sortOrder = sortOrder;
        this.userOriginated = userOriginated;
    }

    /**
     * Gets the sort order list.
     *
     * @return the sort order list
     */
    public List<T> getSortOrder() {
        return sortOrder;
    }

    @Override
    public boolean isUserOriginated() {
        return userOriginated;
    }

    /**
     * Listener for sort order change events.
     *
     * @param <T>
     *            the type of the sorting information
     */
    @FunctionalInterface
    public interface SortListener<T extends SortOrder<?>> extends SerializableEventListener {
        /**
         * Called when the sort order has changed.
         *
         * @param event
         *            the sort order change event
         */
        public void sort(SortEvent<T> event);
    }

    /**
     * The interface for adding and removing listeners for {@link SortEvent
     * SortEvents}.
     *
     * @param <T>
     *            the type of the sorting information
     */
    public interface SortNotifier<T extends SortOrder<?>> extends Serializable {
        /**
         * Adds a sort order change listener that gets notified when the sort
         * order changes.
         *
         * @param listener
         *            the sort order change listener to add
         * @return a registration object for removing the listener
         */
        public Registration addSortListener(SortListener<T> listener);
    }
}
