/*
 * Copyright (C) 2000-2026 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */

package com.vaadin.event;

import java.io.Serializable;

import org.jsoup.Jsoup;
import org.jsoup.safety.Safelist;

import com.vaadin.server.Resource;
import com.vaadin.shared.Registration;
import com.vaadin.shared.ui.ContentMode;

/**
 * Implements the action framework. This class contains subinterfaces for action
 * handling and listing, and for action handler registrations and
 * unregistration.
 *
 * @author Vaadin Ltd.
 * @since 3.0
 */
@SuppressWarnings("serial")
public class Action implements Serializable {

    /**
     * Action title.
     */
    private String caption;

    /**
     * Action icon.
     */
    private Resource icon;

    /**
     * Constructs a new action with the given caption. The caption will be
     * treated as text. Any HTML content in the caption will be stripped, after
     * which the remaining string will be set. If you wish to retain HTML
     * content as part of the caption string, use the
     * {@link #Action(String, ContentMode)} constructor.
     *
     * @see Action#Action(String, ContentMode)
     * @see Action#setCaption(String, ContentMode)
     *
     * @param caption
     *            the caption for the new action.
     */
    public Action(String caption) {
        this(caption, ContentMode.TEXT, null);
    }

    /**
     * Constructs a new action with the given caption and caption content mode.
     * <br>
     * Note that when using the HTML content mode the developer is responsible
     * for ensuring that no harmful HTML is present in the caption string, as
     * the caption may be passed directly into client DOM. Using untrusted input
     * in the caption together with HTML mode can create Cross-site Scripting
     * (XSS) vulnerabilities.
     * 
     * @see Action#setCaption(String, ContentMode)
     * 
     * @param caption
     *            the caption for the new action.
     * @param contentMode
     *            the content mode for the action title. Valid values are
     *            {@link ContentMode#TEXT}, and {@link ContentMode#HTML}. Any
     *            other value causes an IllegalArgumentException to be thrown.
     * @since 8.30
     */
    public Action(String caption, ContentMode contentMode) {
        this(caption, contentMode, null);
    }

    /**
     * Constructs a new action with the given caption string and icon. The
     * caption will be treated as text. Any HTML content in the caption will be
     * stripped, after which the remaining string will be set. If you wish to
     * retain HTML content as part of the caption string, use the
     * {@link #Action(String, ContentMode, Resource)} constructor.
     * 
     * @see Action#Action(String, ContentMode, Resource)
     * @see Action#setCaption(String, ContentMode)
     *
     * @param caption
     *            the caption for the new action.
     * @param icon
     *            the icon for the new action.
     */
    public Action(String caption, Resource icon) {
        this(caption, ContentMode.TEXT, icon);
    }

    /**
     * Constructs a new action with the given caption, caption content mode and
     * icon. <br>
     * Note that when using the HTML content mode the developer is responsible
     * for ensuring that no harmful HTML is present in the caption string, as
     * the caption may be passed directly into client DOM. Using untrusted input
     * in the caption together with HTML mode can create Cross-site Scripting
     * (XSS) vulnerabilities.
     *
     * @see Action#setCaption(String, ContentMode)
     *
     * @param caption
     *            the caption for the new action.
     * @param contentMode
     *            the content mode for the action title. Valid values are
     *            {@link ContentMode#TEXT}, and {@link ContentMode#HTML}. Any
     *            other value causes an IllegalArgumentException to be thrown.
     * @since 8.30
     */
    public Action(String caption, ContentMode contentMode, Resource icon) {
        setCaption(caption, contentMode);
        setIcon(icon);
    }

    /**
     * Returns the action's caption.
     *
     * @return the action's caption as a <code>String</code>.
     */
    public String getCaption() {
        return caption;
    }

    /**
     * Returns the action's icon.
     *
     * @return the action's Icon.
     */
    public Resource getIcon() {
        return icon;
    }

    /**
     * An Action that implements this interface can be added to an
     * Action.Notifier (or NotifierProxy) via the <code>addAction()</code>
     * -method, which in many cases is easier than implementing the
     * Action.Handler interface.<br>
     */
    @FunctionalInterface
    public interface Listener extends Serializable {
        public void handleAction(Object sender, Object target);
    }

    /**
     * Action.Containers implementing this support an easier way of adding
     * single Actions than the more involved Action.Handler. The added actions
     * must be Action.Listeners, thus handling the action themselves.
     */
    public interface Notifier extends Container {
        public <T extends Action & Action.Listener> void addAction(T action);

        public <T extends Action & Action.Listener> void removeAction(T action);
    }

    public interface ShortcutNotifier extends Serializable {
        /**
         * Add a shortcut listener and return a registration object for
         * unregistering it.
         *
         * @param shortcut
         *            listener to add
         * @return registration for unregistering the listener
         * @since 8.0
         */
        public Registration addShortcutListener(ShortcutListener shortcut);

        /**
         * @deprecated As of 8.0, replaced by {@link Registration#remove()} in
         *             the registration object returned from
         *             {@link #addShortcutListener(ShortcutListener)}.
         */
        @Deprecated
        public void removeShortcutListener(ShortcutListener shortcut);
    }

    /**
     * Interface implemented by classes who wish to handle actions.
     *
     * @author Vaadin Ltd.
     * @since 3.0
     */
    public interface Handler extends Serializable {

        /**
         * Gets the list of actions applicable to this handler.
         *
         * @param target
         *            the target handler to list actions for. For item
         *            containers this is the item id.
         * @param sender
         *            the party that would be sending the actions. Most of this
         *            is the action container.
         * @return the list of Action
         */
        public Action[] getActions(Object target, Object sender);

        /**
         * Handles an action for the given target. The handler method may just
         * discard the action if it's not suitable.
         *
         * @param action
         *            the action to be handled.
         * @param sender
         *            the sender of the action. This is most often the action
         *            container.
         * @param target
         *            the target of the action. For item containers this is the
         *            item id.
         */
        public void handleAction(Action action, Object sender, Object target);
    }

    /**
     * Interface implemented by all components where actions can be registered.
     * This means that the components lets others to register as action handlers
     * to it. When the component receives an action targeting its contents it
     * should loop all action handlers registered to it and let them handle the
     * action.
     *
     * @author Vaadin Ltd.
     * @since 3.0
     */
    public interface Container extends Serializable {

        /**
         * Registers a new action handler for this container.
         *
         * @param actionHandler
         *            the new handler to be added.
         */
        public void addActionHandler(Action.Handler actionHandler);

        /**
         * Removes a previously registered action handler for the contents of
         * this container.
         *
         * @param actionHandler
         *            the handler to be removed.
         */
        public void removeActionHandler(Action.Handler actionHandler);
    }

    /**
     * Sets the caption. Note, that any HTML in the caption will be stripped. To
     * allow HTML content in the title, use
     * {@link #setCaption(String, ContentMode)}.
     *
     * @param caption
     *            the caption to set.
     */
    public void setCaption(String caption) {
        setCaption(caption, ContentMode.TEXT);
    }

    /**
     * Sets the caption.
     * <p>
     * If the content mode is {@link ContentMode#TEXT}, any HTML tags in the
     * string will be stripped. If the content mode is {@link ContentMode#HTML},
     * the string is passed as-is. Any other value for {@link ContentMode}
     * causes an {@link IllegalArgumentException} to be thrown.<br>
     * Note that when using the HTML content mode the developer is responsible
     * for ensuring that no harmful HTML is present in the caption string, as
     * the caption may be passed directly into client DOM. Using untrusted input
     * in the caption together with HTML mode can create Cross-site Scripting
     * (XSS) vulnerabilities.
     * </p>
     * 
     * @param caption
     *            the caption to set.
     * @param contentMode
     *            the content mode for the action title. Valid values are
     *            {@link ContentMode#TEXT} and {@link ContentMode#HTML}. Any
     *            other value causes an {@link IllegalArgumentException} to be
     *            thrown.
     * @since 8.30
     */
    public void setCaption(String caption, ContentMode contentMode) {
        if (caption == null) {
            this.caption = null;
            return;
        }

        switch (contentMode) {
        case TEXT:
            // Clean all HTML from text content
            caption = Jsoup.clean(caption, Safelist.none());
            break;
        case HTML:
            // do nothing
            break;
        default:
            throw new IllegalArgumentException(
                    "Unsupported ContentMode value " + contentMode.name());
        }

        this.caption = caption;
    }

    /**
     * Sets the icon.
     *
     * @param icon
     *            the icon to set.
     */
    public void setIcon(Resource icon) {
        this.icon = icon;
    }

}
