/*
 * Copyright (C) 2000-2026 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.osgi.liferay;

import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceObjects;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.log.LogService;
import org.osgi.util.tracker.ServiceTracker;

import com.vaadin.osgi.resources.VaadinResourceService;
import com.vaadin.ui.UI;

/**
 * Initializes a service tracker with {@link PortletUIServiceTrackerCustomizer}
 * to track {@link UI} service registrations.
 * <p>
 * This only applies to Liferay Portal 7+ with OSGi support.
 *
 * @author Sampsa Sohlman
 *
 * @since 8.1
 */
@Component
public class VaadinPortletProvider {

    private ServiceTracker<UI, ServiceObjects<UI>> serviceTracker;
    private PortletUIServiceTrackerCustomizer portletUIServiceTrackerCustomizer;
    private VaadinResourceService vaadinService;
    private LogService logService;

    @Activate
    void activate(BundleContext bundleContext) throws Exception {
        portletUIServiceTrackerCustomizer = new PortletUIServiceTrackerCustomizer(
                vaadinService, logService);
        serviceTracker = new ServiceTracker<UI, ServiceObjects<UI>>(
                bundleContext, UI.class, portletUIServiceTrackerCustomizer);
        serviceTracker.open();
    }
    
    @Reference
    void setVaadinResourceService(VaadinResourceService vaadinService) {
        this.vaadinService = vaadinService;
    }
    
    void unsetVaadinResourceService(VaadinResourceService vaadinService) {
        if(this.vaadinService == vaadinService) {
            this.vaadinService = null;
        }
    }

    @Reference(cardinality = ReferenceCardinality.OPTIONAL)
    void setLogService(LogService logService) {
        this.logService = logService;
    }

    void unsetLogService(LogService logService) {
        if(this.logService == logService) {
            this.logService = null;
        }
    }

    @Deactivate
    void deactivate() {
        if (serviceTracker != null) {
            serviceTracker.close();
            portletUIServiceTrackerCustomizer.cleanPortletRegistrations();
            portletUIServiceTrackerCustomizer = null;
        }
    }
}
