/**
 * Copyright 2000-2024 Vaadin Ltd.
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See  {@literal <https://vaadin.com/commercial-license-and-service-terms>}  for the full
 * license.
 */
package com.vaadin.flow.component.grid.demo;

import java.util.List;
import java.util.Random;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.vaadin.flow.component.grid.demo.GridDemo.Address;
import com.vaadin.flow.component.grid.demo.GridDemo.Person;
import com.vaadin.flow.component.treegrid.demo.TreeGridDemo.PersonWithLevel;

/**
 * Helper class used for generating stable random data for demo purposes.
 *
 * @author Vaadin Ltd.
 *
 */
public class PeopleGenerator extends BeanGenerator {

    private static final AtomicInteger treeIds = new AtomicInteger(0);

    public List<Person> generatePeople(int amount) {
        return IntStream.range(0, amount)
                .mapToObj(index -> createPerson(index + 1))
                .collect(Collectors.toList());
    }

    public List<PersonWithLevel> generatePeopleWithLevels(int amount,
            int level) {
        return IntStream.range(0, amount)
                .mapToObj(index -> createPersonWithLevel(index + 1, level))
                .collect(Collectors.toList());
    }

    private PersonWithLevel createPersonWithLevel(int index, int level) {
        PersonWithLevel person = createPerson(PersonWithLevel::new, index,
                treeIds.getAndIncrement());
        person.setLevel(level);
        return person;
    }

    public Person createPerson(int index) {
        return createPerson(Person::new, index, index);
    }

    private <T extends Person> T createPerson(Supplier<T> constructor,
            int index, int id) {
        boolean isSubscriber = getRandom("subscriber").nextBoolean();

        return createPerson(constructor, "Person " + index, id,
                13 + getRandom("age").nextInt(50), isSubscriber,
                isSubscriber ? generateEmail() : "",
                "Street " + generateChar(getRandom("street"), false),
                1 + getRandom("street").nextInt(50),
                String.valueOf(10000 + getRandom("postalCode").nextInt(8999)));
    }

    private <T extends Person> T createPerson(Supplier<T> constructor,
            String name, int id, int age, boolean subscriber, String email,
            String street, int addressNumber, String postalCode) {
        T person = constructor.get();
        person.setId(id);
        person.setFirstName(name);
        person.setAge(age);
        person.setSubscriber(subscriber);
        person.setEmail(email);

        Address address = new Address();
        address.setStreet(street);
        address.setNumber(addressNumber);
        address.setPostalCode(postalCode);

        person.setAddress(address);

        return person;
    }

    private String generateEmail() {
        StringBuilder builder = new StringBuilder("mail");
        builder.append(generateChar(getRandom("email"), true));
        builder.append(generateChar(getRandom("email"), true));
        builder.append("@example.com");
        return builder.toString();
    }

    private char generateChar(Random random, boolean lowerCase) {
        return ((char) ((lowerCase ? 'a' : 'A') + random.nextInt(26)));
    }

}
