/*
 * Copyright (c) 1997, 2021, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package com.vaadin.featurepack.desktop.layouts;

import java.io.Serializable;

import com.vaadin.flow.component.AttachEvent;
import com.vaadin.flow.component.Component;

/**
 * Flow layout arranges components in a directional flow.
 *
 * @author Vaadin Ltd
 */
public class BaseFlowLayout implements Serializable {

    protected int horizontalGap = 0;
    protected int verticalGap = 0;

    private int alignment = CENTER;

    /**
     * This value indicates that each row of components should be centered.
     */
    public static final int CENTER = 0;

    /**
     * This value indicates that each row of components should be justified to
     * the start of the container (for left-to-right direction, it is the left
     * edge).
     */
    public static final int LEADING = 1;

    /**
     * This value indicates that each row of components should be justified to
     * the end of the container (for left-to-right direction, it is the right
     * edge).
     */
    public static final int TRAILING = 2;

    /**
     * Constructs a new {@code FlowLayout} with a centered alignment and a
     * default 5-pixels horizontal and vertical gap.
     */
    public BaseFlowLayout() {
        this(CENTER, 5, 5);
    }

    /**
     * Constructs a new {@code FlowLayout} with the specified alignment and a
     * default 5-pixels horizontal and vertical gap.
     *
     * @param align
     *            the alignment value
     */
    public BaseFlowLayout(int align) {
        this(align, 5, 5);
    }

    /**
     * Constructs a new {@code FlowLayout} with the specified alignment and the
     * indicated horizontal and vertical gaps.
     *
     * @param align
     *            the alignment value
     * @param hgap
     *            the horizontal gap between components and between the
     *            components and the borders of the container
     * @param vgap
     *            the vertical gap between components and between the components
     *            and the borders of the container
     */
    public BaseFlowLayout(int align, int hgap, int vgap) {
        setHorizontalGap(hgap);
        setVerticalGap(vgap);
    }

    /**
     * Gets the alignment for this layout. Possible values are
     * {@code FlowLayout.LEADING}, {@code FlowLayout.CENTER}, or
     * {@code FlowLayout.TRAILING}.
     */
    public int getAlignment() {
        return alignment;
    }

    /**
     * Sets the alignment for this layout. Possible values are
     * {@code FlowLayout.LEADING}, {@code FlowLayout.CENTER}, or
     * {@code FlowLayout.TRAILING}.
     *
     * @param align
     *            the alignment value
     */
    protected void setAlignment(int align, Component container) {
        alignment = align;

        updateAlignment(container);
    }

    protected void updateAlignment(Component container) {
        String justify;
        switch (alignment) {
        case LEADING:
            justify = "flex-start";
            break;
        case TRAILING:
            justify = "flex-end";
            break;
        default:
            justify = "center";
            break;
        }

        if (container != null) {
            container.getStyle().set("--vfp-justify-content", justify);
        }
    }

    protected void setHorizontalGap(int horizontalGap) {
        this.horizontalGap = horizontalGap;
    }

    protected void setVerticalGap(int verticalGap) {
        this.verticalGap = verticalGap;
    }

    protected int getHorizontalGap() {
        return horizontalGap;
    }

    protected int getVerticalGap() {
        return verticalGap;
    }
}
