/*
 * Copyright (c) 1997, 2021, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package com.vaadin.featurepack.desktop.layouts;

import java.io.Serializable;

import com.vaadin.featurepack.desktop.geometry.Dimension;
import com.vaadin.flow.component.Component;

public interface LayoutManager extends Serializable {
    /**
     * Adds the specified component to the layout, using the specified
     * constraint object.
     *
     * @param comp
     *            the component to be added
     * @param constraints
     *            where/how the component is added to the layout.
     */
    void addLayoutComponent(Component comp, Object constraints);

    /**
     * Calculates the maximum size dimensions for the specified container, given
     * the components it contains.
     *
     * @see java.awt.Component#getMaximumSize
     * @see LayoutManager
     * @param target
     *            the target container
     * @return the maximum size of the container
     */
    public Dimension maximumLayoutSize(Component container);

    /**
     * Returns the alignment along the x axis. This specifies how the component
     * would like to be aligned relative to other components. The value should
     * be a number between 0 and 1 where 0 represents alignment along the
     * origin, 1 is aligned the furthest away from the origin, 0.5 is centered,
     * etc.
     *
     * @param target
     *            the target container
     * @return the x-axis alignment preference
     */
    public float getLayoutAlignmentX(Component container);

    /**
     * Returns the alignment along the y axis. This specifies how the component
     * would like to be aligned relative to other components. The value should
     * be a number between 0 and 1 where 0 represents alignment along the
     * origin, 1 is aligned the furthest away from the origin, 0.5 is centered,
     * etc.
     *
     * @param target
     *            the target container
     * @return the y-axis alignment preference
     */
    public float getLayoutAlignmentY(Component container);

    /**
     * Invalidates the layout, indicating that if the layout manager has cached
     * information it should be discarded.
     *
     * @param target
     *            the target container
     */
    public void invalidateLayout(Component container);

    /**
     * If the layout manager uses a per-component string, adds the component
     * {@code comp} to the layout, associating it with the string specified by
     * {@code name}.
     *
     * @param name
     *            the string to be associated with the component
     * @param comp
     *            the component to be added
     */
    void addLayoutComponent(String name, Component comp);

    /**
     * Removes the specified component from the layout.
     *
     * @param comp
     *            the component to be removed
     */
    void removeLayoutComponent(Component container);

    /**
     * Calculates the preferred size dimensions for the specified container,
     * given the components it contains.
     *
     * @param parent
     *            the container to be laid out
     * @return the preferred dimension for the container
     *
     * @see #minimumLayoutSize
     */
    Dimension preferredLayoutSize(Component container);

    /**
     * Calculates the minimum size dimensions for the specified container, given
     * the components it contains.
     *
     * @param parent
     *            the component to be laid out
     * @return the minimum dimension for the container
     *
     * @see #preferredLayoutSize
     */
    Dimension minimumLayoutSize(Component container);

    /**
     * Lays out the specified container.
     *
     * @param parent
     *            the container to be laid out
     */
    void layoutContainer(Component container);

}
