/*
 * Copyright (c) 1997, 2021, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package com.vaadin.featurepack.desktop.layouts;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import com.vaadin.flow.component.Component;

/**
 * Card layout treats each component in the container as a card. Only one card
 * is visible at a time, and the container acts as a stack of cards.
 *
 * @author Vaadin Ltd
 */
public class BaseCardLayout implements Serializable {

    protected int currentCard = 0;
    protected List<Component> cards = new ArrayList<>();

    /**
     * Shows the first card and hides all the others.
     */
    public void first() {
        for (int i = 0; i < cards.size(); i++) {
            Component card = cards.get(i);
            if (card.isVisible()) {
                card.setVisible(false);
                break;
            }
        }

        if (cards.size() > 0) {
            currentCard = 0;
            cards.get(currentCard).setVisible(true);
        }
    }

    /**
     * Shows the last card and hides all the others.
     */
    public void last() {
        for (int i = 0; i < cards.size(); i++) {
            Component card = cards.get(i);
            if (card.isVisible()) {
                card.setVisible(false);
                break;
            }
        }

        if (cards.size() > 0) {
            currentCard = cards.size() - 1;
            cards.get(currentCard).setVisible(true);
        }
    }

    /**
     * Shows the previous card and hides all the others. If the currently
     * visible card is the first one, the last card is shown.
     */
    public void previous() {
        int count = cards.size();
        for (int i = 0; i < count; i++) {
            Component card = cards.get(i);
            if (card.isVisible()) {
                card.setVisible(false);
                currentCard = ((i > 0) ? i - 1 : count - 1);
                cards.get(currentCard).setVisible(true);
                return;
            }
        }
    }

    /**
     * Shows the next card and hides all the others. If the currently visible
     * card is the last one, the first card is shown.
     */
    public void next() {
        int count = cards.size();
        for (int i = 0; i < count; i++) {
            Component card = cards.get(i);
            if (card.isVisible()) {
                card.setVisible(false);
                currentCard = (i + 1) % count;
                cards.get(currentCard).setVisible(true);
                return;
            }
        }
    }

    /**
     * Shows the component that was added to this layout with the specified
     * name, using {@link #setConstraints(Component, Object)}. If no such
     * component exists, then nothing happens.
     *
     * @param name
     *            the component name
     */
    public void show(String name) {
        Component next = null;
        for (int i = 0; i < cards.size(); i++) {
            Component card = cards.get(i);
            if (card.getElement().getAttribute("vfp-card-layout-name")
                    .equals(name)) {
                next = card;
                break;
            }
        }

        if ((next != null) && !next.isVisible()) {
            for (int i = 0; i < cards.size(); i++) {
                Component card = cards.get(i);
                if (card.isVisible()) {
                    card.setVisible(false);
                    break;
                }
            }

            next.setVisible(true);
        }
    }

    public List<Component> getCards() {
        return cards;
    }

    public int getCurrentCard() {
        return currentCard;
    }

    public void setCurrentCard(int currentCard) {
        this.currentCard = currentCard;
    }
}
