/*
 * Copyright (c) 1997, 2021, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package com.vaadin.featurepack.desktop.geometry;

import java.io.Serializable;
import java.util.Objects;

/**
 * A class that represents a rectangle in 2D space.
 *
 * @author Vaadin Ltd
 */
public class Rectangle implements Serializable {
    private int x;
    private int y;
    private int width;
    private int height;

    /**
     * Creates a new rectangle with coordinates (0, 0) and size 0x0.
     */
    public Rectangle() {
        this(0, 0, 0, 0);
    }

    /**
     * Creates a new rectangle with the given coordinates and size.
     *
     * @param x
     *            the x coordinate
     * @param y
     *            the y coordinate
     * @param width
     *            the width
     * @param height
     *            the height
     */
    public Rectangle(int x, int y, int width, int height) {
        this.x = x;
        this.y = y;
        this.width = width;
        this.height = height;
    }

    /**
     * Creates a new rectangle from a {@code Point} and {@code Dimension}
     * objects.
     *
     * @param p
     *            the Point object representing the location
     * @param d
     *            the Dimension object representing the size
     */
    public Rectangle(Point p, Dimension d) {
        this(p.getX(), p.getY(), d.getWidth(), d.getHeight());
    }

    /**
     * Creates a new rectangle clone of another Rectangle
     *
     * @param Rectangle
     *            the Rectangle to clone
     */
    public Rectangle(Rectangle r) {
        this(r.getX(), r.getY(), r.getWidth(), r.getHeight());
    }

    /**
     * Returns the location of this rectangle.
     *
     * @return the location
     */
    public Point getLocation() {
        return new Point(x, y);
    }

    /**
     * Returns the size of this rectangle.
     *
     * @return the size
     */
    public Dimension getSize() {
        return new Dimension(width, height);
    }

    /**
     * Returns the x coordinate of the left side of the rectangle.
     *
     * @return the x coordinate
     */
    public int getX() {
        return x;
    }

    /**
     * Returns the y coordinate of the top side of the rectangle.
     *
     * @return the y coordinate
     */
    public int getY() {
        return y;
    }

    /**
     * Returns the width of the rectangle.
     *
     * @return the width
     */
    public int getWidth() {
        return width;
    }

    /**
     * Returns the height of the rectangle.
     *
     * @return the height
     */
    public int getHeight() {
        return height;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj instanceof Rectangle rectangle) {
            return (getX() == rectangle.getX()) && (getY() == rectangle.getY())
                    && (getWidth() == rectangle.getWidth())
                    && (getHeight() == rectangle.getHeight());
        }
        return false;
    }

    @Override
    public int hashCode() {
        return Objects.hash(x, y, width, height);
    }
}
