/*
 * Copyright (c) 1997, 2021, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package com.vaadin.featurepack.desktop.geometry;

import java.io.Serializable;
import java.util.Objects;

/**
 * A class that represents the external padding of a container. It specifies how
 * much extra space should be left around the container.
 *
 * @author Vaadin Ltd
 */
public class Insets implements Cloneable, Serializable {
    /**
     * The inset from the top.
     */
    public int top;

    /**
     * The inset from the left.
     */
    public int left;

    /**
     * The inset from the bottom.
     */
    public int bottom;

    /**
     * The inset from the right.
     */
    public int right;

    /**
     * Creates a new {@code Insets} object with the specified top, left, bottom,
     * and right insets.
     *
     * @param top
     *            the inset from the top
     * @param left
     *            the inset from the left
     * @param bottom
     *            the inset from the bottom
     * @param right
     *            the inset from the right
     */
    public Insets(int top, int left, int bottom, int right) {
        this.top = top;
        this.left = left;
        this.bottom = bottom;
        this.right = right;
    }

    /**
     * Sets the insets to the specified values.
     *
     * @param top
     *            the inset from the top
     * @param left
     *            the inset from the left
     * @param bottom
     *            the inset from the bottom
     * @param right
     *            the inset from the right
     */
    public void set(int top, int left, int bottom, int right) {
        this.top = top;
        this.left = left;
        this.bottom = bottom;
        this.right = right;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj instanceof Insets insets) {
            return top == insets.top && left == insets.left
                    && bottom == insets.bottom && right == insets.right;
        }
        return false;
    }

    @Override
    public int hashCode() {
        return Objects.hash(top, left, bottom, right);
    }

    @Override
    public Object clone() {
        try {
            return super.clone();
        } catch (CloneNotSupportedException e) {
            // this shouldn't happen, since we are Cloneable
            throw new InternalError(e);
        }
    }

}
