/*
 * Copyright (c) 2002-2015 JGoodies Software GmbH. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of JGoodies Software GmbH nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.vaadin.featurepack.desktop.layouts.form;

import java.io.Serializable;
import java.util.Locale;

/**
 * Consists only of static methods that create constant sizes as required by the
 * {@link FormLayout}.
 */
public final class Sizes implements Serializable {

    private Sizes() {
        // Suppresses default constructor, prevents instantiation.
    }

    /**
     * Creates and returns an instance of {@code ConstantSize} from the given
     * encoded size and unit description.
     *
     * @param encodedValueAndUnit
     *            value and unit in string representation
     * @return a {@code ConstantSize} for the given value and unit
     */
    public static ConstantSize constant(String encodedValueAndUnit) {
        String lowerCase = encodedValueAndUnit.toLowerCase(Locale.ENGLISH);
        String trimmed = lowerCase.trim();
        return ConstantSize.decode(trimmed);
    }

    /**
     * Creates and returns a ConstantSize for the specified value in horizontal
     * dialog units.
     *
     * @param value
     *            size value in horizontal dialog units
     * @return the associated {@code ConstantSize}
     */
    public static ConstantSize dluX(int value) {
        return ConstantSize.dlu(value);
    }

    /**
     * Creates and returns a ConstantSize for the specified value in vertical
     * dialog units.
     *
     * @param value
     *            size value in vertical dialog units
     * @return the associated {@code ConstantSize}
     */
    public static ConstantSize dluY(int value) {
        return ConstantSize.dlu(value);
    }

    /**
     * Creates and returns a ConstantSize for the specified pixel value.
     *
     * @param value
     *            value in pixel
     * @return the associated {@code ConstantSize}
     */
    public static ConstantSize pixel(int value) {
        return new ConstantSize(value, Unit.PIXEL);
    }
}
