/*
 * Copyright (c) 2002-2015 JGoodies Software GmbH. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of JGoodies Software GmbH nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.vaadin.featurepack.desktop.layouts.form;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;

/**
 * Specifies columns in {@link FormLayout} by their default alignment, size and
 * resizing behavior.
 */
public final class ColumnSpec extends FormSpec {
    /**
     * By default put components in the left.
     */
    public static final DefaultAlignment LEFT = FormSpec.LEFT_ALIGN;

    /**
     * By default put the components in the center.
     */
    public static final DefaultAlignment CENTER = FormSpec.CENTER_ALIGN;

    /**
     * By default put components in the right.
     */
    public static final DefaultAlignment RIGHT = FormSpec.RIGHT_ALIGN;

    /**
     * By default fill the component into the column.
     */
    public static final DefaultAlignment FILL = FormSpec.FILL_ALIGN;

    /**
     * Unless overridden the default alignment for a column is FILL.
     */
    public static final DefaultAlignment DEFAULT = FILL;

    /**
     * Maps encoded column specifications to ColumnSpec instances.
     */
    private static final Map<String, ColumnSpec> CACHE = new HashMap<>();

    /**
     * Constructs a ColumnSpec for the given default alignment, size and resize
     * weight. The resize weight must be a non-negative double; you can use
     * {@code NO_GROW} as a convenience value for no resize.
     *
     * @param defaultAlignment
     *            the column's default alignment
     * @param size
     *            constant, component size or bounded size
     * @param resizeWeight
     *            the column's non-negative resize weight
     *
     * @throws NullPointerException
     *             if the {@code size} is {@code null}
     * @throws IllegalArgumentException
     *             if the size is invalid or the {@code resizeWeight} is
     *             negative
     */
    public ColumnSpec(DefaultAlignment defaultAlignment, Size size,
            double resizeWeight) {
        super(defaultAlignment, size, resizeWeight);
    }

    /**
     * Constructs a ColumnSpec for the given size using the default alignment,
     * and no resizing.
     *
     * @param size
     *            constant size, component size, or bounded size
     * @throws IllegalArgumentException
     *             if the size is invalid
     */
    public ColumnSpec(Size size) {
        super(DEFAULT, size, NO_GROW);
    }

    /**
     * Constructs a ColumnSpec from the specified encoded description. The
     * description will be parsed to set initial values.
     *
     * @param encodedDescription
     *            the encoded description
     */
    private ColumnSpec(String encodedDescription) {
        super(DEFAULT, encodedDescription);
    }

    /**
     * Creates and returns a {@link ColumnSpec} that represents a gap with the
     * specified {@link ConstantSize}.
     *
     * @param gapWidth
     *            specifies the gap width
     * @return a ColumnSpec that describes a horizontal gap
     *
     * @throws NullPointerException
     *             if {@code gapWidth} is {@code null}
     */
    public static ColumnSpec createGap(ConstantSize gapWidth) {
        return new ColumnSpec(DEFAULT, gapWidth, FormSpec.NO_GROW);
    }

    /**
     * Parses the encoded column specifications and returns a ColumnSpec object
     * that represents the string.
     *
     * @param encodedColumnSpec
     *            the encoded column specification
     *
     * @return a ColumnSpec instance for the given specification
     * @throws NullPointerException
     *             if {@code encodedColumnSpec} is {@code null}
     */
    public static ColumnSpec decode(String encodedColumnSpec) {
        Objects.requireNonNull(encodedColumnSpec,
                "Column specification cannot be null");
        String trimmed = encodedColumnSpec.trim();
        return decodeExpanded(trimmed.toLowerCase(Locale.ENGLISH));
    }

    /**
     * Decodes a trimmed, lower case column spec. Called by the public
     * ColumnSpec factory methods. Looks up and returns the ColumnSpec object
     * from the cache - if any, or constructs and returns a new ColumnSpec
     * instance.
     *
     * @param expandedTrimmedLowerCaseSpec
     *            the encoded column specification
     * @return a ColumnSpec for the given encoded column spec
     */
    static ColumnSpec decodeExpanded(String expandedTrimmedLowerCaseSpec) {
        ColumnSpec spec = CACHE.get(expandedTrimmedLowerCaseSpec);
        if (spec == null) {
            spec = new ColumnSpec(expandedTrimmedLowerCaseSpec);
            CACHE.put(expandedTrimmedLowerCaseSpec, spec);
        }
        return spec;
    }

    /**
     * Parses and splits encoded column specifications and returns an array of
     * ColumnSpec objects.
     *
     * @param encodedColumnSpecs
     *            comma separated encoded column specifications
     * @return an array of decoded column specifications
     * @throws NullPointerException
     *             if {@code encodedColumnSpecs} is {@code null}
     */
    public static ColumnSpec[] decodeSpecs(String encodedColumnSpecs) {
        Objects.requireNonNull(encodedColumnSpecs,
                "Column specification cannot be null");
        return FormSpecParser.parseColumnSpecs(encodedColumnSpecs);
    }

    /**
     * Returns if this is a horizontal specification (vs. vertical). Used to
     * distinct between horizontal and vertical dialog units, which have
     * different conversion factors.
     *
     * @return always {@code true} (for horizontal)
     */
    @Override
    protected boolean isHorizontal() {
        return true;
    }
}
