/*
 * Copyright (c) 2002-2015 JGoodies Software GmbH. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of JGoodies Software GmbH nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.vaadin.featurepack.desktop.layouts.form;

import java.io.Serializable;
import java.util.Locale;
import java.util.Objects;
import java.util.StringTokenizer;

/**
 * Defines constraints for placing components inside the {@code FormLayout}.
 * Constraints describe the components display area: gridX, gridY, grid width
 * (column span), and grid height (row span). Cell constraints need to be set
 * using {@link FormLayout#setConstraints(Component, Object)}.
 * <p>
 * Most methods return <em>this</em> object to enable method chaining. Note,
 * {@code FormLayout} implicitly clones the cell constraints instance when
 * applying it to specific components.
 */
public class CellConstraints implements Cloneable, Serializable {
    /**
     * Use the column's or row's default alignment.
     */
    public static final Alignment DEFAULT = new Alignment("default",
            Alignment.BOTH);

    /**
     * Fill the cell either horizontally or vertically.
     */
    public static final Alignment FILL = new Alignment("fill", Alignment.BOTH);

    /**
     * Put the component in the left.
     */
    public static final Alignment LEFT = new Alignment("left",
            Alignment.HORIZONTAL);

    /**
     * Put the component in the right.
     */
    public static final Alignment RIGHT = new Alignment("right",
            Alignment.HORIZONTAL);

    /**
     * Put the component in the center.
     */
    public static final Alignment CENTER = new Alignment("center",
            Alignment.BOTH);

    /**
     * Put the component in the top.
     */
    public static final Alignment TOP = new Alignment("top",
            Alignment.VERTICAL);

    /**
     * Put the component in the bottom.
     */
    public static final Alignment BOTTOM = new Alignment("bottom",
            Alignment.VERTICAL);

    /**
     * An array of all enumeration values used to canonicalize deserialized
     * alignments.
     */
    private static final Alignment[] VALUES = { DEFAULT, FILL, LEFT, RIGHT,
            CENTER, TOP, BOTTOM };

    /**
     * Describes the component's horizontal grid origin (starts at 1).
     */
    public int gridX;

    /**
     * Describes the component's vertical grid origin (starts at 1).
     */
    public int gridY;

    /**
     * Describes the component's horizontal grid extend (number of cells).
     */
    public int gridWidth;

    /**
     * Describes the component's vertical grid extent (number of cells).
     */
    public int gridHeight;

    /**
     * Describes the component's horizontal alignment.
     */
    public Alignment hAlign;

    /**
     * Describes the component's vertical alignment.
     */
    public Alignment vAlign;

    /**
     * Constructs a default instance of {@code CellConstraints}.
     */
    public CellConstraints() {
        this(1, 1);
    }

    /**
     * Constructs an instance of {@code CellConstraints} for the given cell
     * position.
     *
     * @param gridX
     *            the component's horizontal grid origin
     * @param gridY
     *            the component's vertical grid origin
     */
    public CellConstraints(int gridX, int gridY) {
        this(gridX, gridY, 1, 1);
    }

    /**
     * Constructs an instance of {@code CellConstraints} for the given cell
     * position and alignment.
     *
     * @param gridX
     *            the component's horizontal grid origin
     * @param gridY
     *            the component's vertical grid origin
     * @param hAlign
     *            the component's horizontal alignment
     * @param vAlign
     *            the component's vertical alignment
     */
    public CellConstraints(int gridX, int gridY, Alignment hAlign,
            Alignment vAlign) {
        this(gridX, gridY, 1, 1, hAlign, vAlign);
    }

    /**
     * Constructs an instance of {@code CellConstraints} for the given cell
     * position and size.
     *
     * @param gridX
     *            the component's horizontal grid origin
     * @param gridY
     *            the component's vertical grid origin
     * @param gridWidth
     *            the component's horizontal extent
     * @param gridHeight
     *            the component's vertical extent
     */
    public CellConstraints(int gridX, int gridY, int gridWidth,
            int gridHeight) {
        this(gridX, gridY, gridWidth, gridHeight, DEFAULT, DEFAULT);
    }

    /**
     * Constructs an instance of {@code CellConstraints} for the given cell
     * position, size and alignment.
     *
     * @param gridX
     *            the component's horizontal grid origin
     * @param gridY
     *            the component's vertical grid origin
     * @param gridWidth
     *            the component's horizontal extent
     * @param gridHeight
     *            the component's vertical extent
     * @param hAlign
     *            the component's horizontal alignment
     * @param vAlign
     *            the component's vertical alignment
     * @throws IndexOutOfBoundsException
     *             if the grid origin or extent is negative
     * @throws NullPointerException
     *             if the horizontal or vertical alignment is null
     * @throws IllegalArgumentException
     *             if an alignment orientation is invalid
     */
    public CellConstraints(int gridX, int gridY, int gridWidth, int gridHeight,
            Alignment hAlign, Alignment vAlign) {
        Objects.requireNonNull(hAlign, "Horizontal alignment must not be null");
        Objects.requireNonNull(vAlign, "Vertical alignment must not be null");

        this.gridX = gridX;
        this.gridY = gridY;
        this.gridWidth = gridWidth;
        this.gridHeight = gridHeight;
        this.hAlign = hAlign;
        this.vAlign = vAlign;

        if (gridX <= 0) {
            throw new IndexOutOfBoundsException(
                    "The grid x must be a positive number.");
        }

        if (gridY <= 0) {
            throw new IndexOutOfBoundsException(
                    "The grid y must be a positive number.");
        }

        if (gridWidth <= 0) {
            throw new IndexOutOfBoundsException(
                    "The grid width must be a positive number.");
        }

        if (gridHeight <= 0) {
            throw new IndexOutOfBoundsException(
                    "The grid height must be a positive number.");
        }

        ensureValidOrientations(hAlign, vAlign);
    }

    /**
     * Constructs an instance of {@code CellConstraints} from the given encoded
     * string properties.
     * <p>
     * <strong>Examples:</strong>
     *
     * <pre>
     * new CellConstraints("1, 3");
     * new CellConstraints("1, 3, left, bottom");
     * new CellConstraints("1, 3, 2, 1, left, bottom");
     * new CellConstraints("1, 3, 2, 1, l, b");
     * </pre>
     *
     * @param encodedConstraints
     *            the constraints encoded as string
     */
    public CellConstraints(String encodedConstraints) {
        this();
        initFromConstraints(encodedConstraints);
    }

    /**
     * Sets column and row origins; sets width and height to 1.
     *
     * @param col
     *            the new column index
     * @param row
     *            the new row index
     * @return this
     */
    public CellConstraints xy(int col, int row) {
        return xywh(col, row, 1, 1);
    }

    /**
     * Sets column and row origins; sets width and height to 1; decodes
     * horizontal and vertical alignments from the given string.
     *
     * @param col
     *            the new column index
     * @param row
     *            the new row index
     * @param encodedAlignments
     *            describes the horizontal and vertical alignments
     * @return this
     *
     * @throws IllegalArgumentException
     *             if an alignment orientation is invalid
     */
    public CellConstraints xy(int col, int row, String encodedAlignments) {
        return xywh(col, row, 1, 1, encodedAlignments);
    }

    /**
     * Sets the column and row origins; sets width and height to 1; set
     * horizontal and vertical alignment using the specified objects.
     *
     * @param col
     *            the new column index
     * @param row
     *            the new row index
     * @param colAlign
     *            horizontal component alignment
     * @param rowAlign
     *            vertical component alignment
     * @return this
     */
    public CellConstraints xy(int col, int row, Alignment colAlign,
            Alignment rowAlign) {
        return xywh(col, row, 1, 1, colAlign, rowAlign);
    }

    /**
     * Sets the column, row, and width; uses a height (row span) of 1.
     *
     * @param col
     *            the new column index
     * @param row
     *            the new row index
     * @param colSpan
     *            the column span or grid width
     * @return this
     */
    public CellConstraints xyw(int col, int row, int colSpan) {
        return xywh(col, row, colSpan, 1, DEFAULT, DEFAULT);
    }

    /**
     * Sets the column, row, width, and height; decodes the horizontal and
     * vertical alignments from the given string. Uses a height (row span) of 1.
     * <p>
     *
     * <strong>Examples:</strong>
     *
     * <pre>
     * cc.xyw(1, 3, 7, "left, bottom");
     * cc.xyw(1, 3, 7, "l, b");
     * cc.xyw(1, 3, 2, "center, fill");
     * cc.xyw(1, 3, 2, "c, f");
     * </pre>
     *
     * @param col
     *            the new column index
     * @param row
     *            the new row index
     * @param colSpan
     *            the column span or grid width
     * @param encodedAlignments
     *            describes the horizontal and vertical alignments
     * @return this
     * @throws IllegalArgumentException
     *             if an alignment orientation is invalid
     */
    public CellConstraints xyw(int col, int row, int colSpan,
            String encodedAlignments) {
        return xywh(col, row, colSpan, 1, encodedAlignments);
    }

    /**
     * Sets the column, row, width, and height; sets the horizontal and vertical
     * alignment using the specified objects. Uses a height (row span) of 1.
     * <p>
     *
     * <strong>Examples:</strong>
     *
     * <pre>
     * cc.xyw(1, 3, 2, CellConstraints.LEFT, CellConstraints.BOTTOM);
     * cc.xyw(1, 3, 7, CellConstraints.CENTER, CellConstraints.FILL);
     * </pre>
     *
     * @param col
     *            the new column index
     * @param row
     *            the new row index
     * @param colSpan
     *            the column span or grid width
     * @param colAlign
     *            horizontal component alignment
     * @param rowAlign
     *            vertical component alignment
     * @return this
     * @throws IllegalArgumentException
     *             if an alignment orientation is invalid
     */
    public CellConstraints xyw(int col, int row, int colSpan,
            Alignment colAlign, Alignment rowAlign) {
        return xywh(col, row, colSpan, 1, colAlign, rowAlign);
    }

    /**
     * Sets the column, row, width, and height.
     *
     * @param col
     *            the new column index
     * @param row
     *            the new row index
     * @param colSpan
     *            the column span or grid width
     * @param rowSpan
     *            the row span or grid height
     * @return this
     */
    public CellConstraints xywh(int col, int row, int colSpan, int rowSpan) {
        return xywh(col, row, colSpan, rowSpan, DEFAULT, DEFAULT);
    }

    /**
     * Sets the column, row, width, and height; decodes the horizontal and
     * vertical alignments from the given string.
     * <p>
     *
     * @param col
     *            the new column index
     * @param row
     *            the new row index
     * @param colSpan
     *            the column span or grid width
     * @param rowSpan
     *            the row span or grid height
     * @param encodedAlignments
     *            describes the horizontal and vertical alignments
     * @return this
     * @throws IllegalArgumentException
     *             if an alignment orientation is invalid
     */
    public CellConstraints xywh(int col, int row, int colSpan, int rowSpan,
            String encodedAlignments) {
        CellConstraints result = xywh(col, row, colSpan, rowSpan);
        result.setAlignments(encodedAlignments, true);
        return result;
    }

    /**
     * Sets the column, row, width, and height; sets the horizontal and vertical
     * alignment using the specified alignment objects.
     * <p>
     *
     * @param col
     *            the new column index
     * @param row
     *            the new row index
     * @param colSpan
     *            the column span or grid width
     * @param rowSpan
     *            the row span or grid height
     * @param colAlign
     *            horizontal component alignment
     * @param rowAlign
     *            vertical component alignment
     * @return this
     * @throws IllegalArgumentException
     *             if an alignment orientation is invalid
     */
    public CellConstraints xywh(int col, int row, int colSpan, int rowSpan,
            Alignment colAlign, Alignment rowAlign) {
        this.gridX = col;
        this.gridY = row;
        this.gridWidth = colSpan;
        this.gridHeight = rowSpan;
        this.hAlign = colAlign;
        this.vAlign = rowAlign;
        ensureValidOrientations(hAlign, vAlign);
        return this;
    }

    /**
     * Decodes and returns the grid bounds and alignments for this constraints
     * as an array of six integers. The string representation is a comma
     * separated sequence, one of
     *
     * <pre>
     * "x, y"
     * "x, y, w, h"
     * "x, y, hAlign, vAlign"
     * "x, y, w, h, hAlign, vAlign"
     * </pre>
     *
     * @param encodedConstraints
     *            represents horizontal and vertical alignment
     *
     * @throws IllegalArgumentException
     *             if the encoded constraints do not follow the constraint
     *             syntax
     */
    @SuppressWarnings("null")
    private void initFromConstraints(String encodedConstraints) {
        StringTokenizer tokenizer = new StringTokenizer(encodedConstraints,
                " ,");
        int argCount = tokenizer.countTokens();
        if (argCount != 2 && argCount != 4 && argCount != 6) {
            throw new IllegalArgumentException(
                    "You must provide 2, 4 or 6 arguments.");
        }
        Integer nextInt = decodeInt(tokenizer.nextToken());
        if (nextInt == null) {
            throw new IllegalArgumentException(
                    "First cell constraint element must be a number.");
        }
        gridX = nextInt.intValue();
        if (gridX <= 0) {
            throw new IllegalArgumentException(
                    "The grid X must be a positive number.");
        }
        nextInt = decodeInt(tokenizer.nextToken());
        if (nextInt == null) {
            throw new IllegalArgumentException(
                    "Second cell constraint element must be a number.");
        }
        gridY = nextInt.intValue();
        if (gridY <= 0) {
            throw new IllegalArgumentException(
                    "The grid Y must be a positive number.");
        }
        if (!tokenizer.hasMoreTokens()) {
            return;
        }

        String token = tokenizer.nextToken();
        nextInt = decodeInt(token);
        if (nextInt != null) {
            // Case: "x, y, w, h" or
            // "x, y, w, h, hAlign, vAlign"
            gridWidth = nextInt.intValue();
            if (gridWidth <= 0) {
                throw new IndexOutOfBoundsException(
                        "The grid width must be a positive number.");
            }
            nextInt = decodeInt(tokenizer.nextToken());
            if (nextInt == null) {
                throw new IllegalArgumentException(
                        "Fourth cell constraint element must be like third.");
            }
            gridHeight = nextInt.intValue();
            if (gridHeight <= 0) {
                throw new IndexOutOfBoundsException(
                        "The grid height must be a positive number.");
            }

            if (!tokenizer.hasMoreTokens()) {
                return;
            }
            token = tokenizer.nextToken();
        }

        hAlign = decodeAlignment(token);
        vAlign = decodeAlignment(tokenizer.nextToken());
        ensureValidOrientations(hAlign, vAlign);
    }

    /**
     * Decodes a string description for the horizontal and vertical alignment
     * and sets this CellConstraints' alignment values. If the boolean is
     * {@code true} the horizontal alignment is the first token, and the
     * vertical alignment is the second token. if the boolean is {@code false}
     * the vertical alignment comes first.
     * <p>
     * Valid horizontal alignments are: left, center, right, default, and fill.
     * Valid vertical alignments are: top, center, bottom, default, and fill.
     *
     * @param encodedAlignments
     *            represents horizontal and vertical alignment
     * @throws IllegalArgumentException
     *             if an alignment orientation is invalid
     */
    private void setAlignments(String encodedAlignments,
            boolean horizontalThenVertical) {
        StringTokenizer tokenizer = new StringTokenizer(encodedAlignments,
                " ,");
        Alignment first = decodeAlignment(tokenizer.nextToken());
        Alignment second = decodeAlignment(tokenizer.nextToken());
        hAlign = horizontalThenVertical ? first : second;
        vAlign = horizontalThenVertical ? second : first;
        ensureValidOrientations(hAlign, vAlign);
    }

    /**
     * Decodes an integer string representation and returns the associated
     * Integer or null in case of an invalid number format.
     *
     * @param token
     *            the encoded integer
     * @return the decoded Integer or null
     */
    private static Integer decodeInt(String token) {
        try {
            return Integer.decode(token);
        } catch (NumberFormatException e) {
            return null;
        }
    }

    /**
     * Parses an alignment string description and returns the corresponding
     * alignment value.
     *
     * @param encodedAlignment
     *            the encoded alignment
     * @return the associated {@code Alignment} instance
     */
    private static Alignment decodeAlignment(String encodedAlignment) {
        return Alignment.valueOf(encodedAlignment);
    }

    /**
     * Checks and verifies that the provided horizontal alignment is horizontal
     * and the provided vertical alignment is vertical.
     *
     * @param horizontalAlignment
     *            the horizontal alignment
     * @param verticalAlignment
     *            the vertical alignment
     * @throws IllegalArgumentException
     *             if an alignment is invalid
     */
    private static void ensureValidOrientations(Alignment horizontalAlignment,
            Alignment verticalAlignment) {
        if (!horizontalAlignment.isHorizontal()) {
            throw new IllegalArgumentException(
                    "The horizontal alignment must be one of: left, center, right, fill, default.");
        }

        if (!verticalAlignment.isVertical()) {
            throw new IllegalArgumentException(
                    "The vertical alignment must be one of: top, center, bottom, fill, default.");
        }
    }

    /**
     * Creates a copy of this cell constraints object.
     *
     * @return a copy of this cell constraints object
     */
    @Override
    public CellConstraints clone() {
        try {
            CellConstraints c = (CellConstraints) super.clone();
            return c;
        } catch (CloneNotSupportedException e) {
            // This shouldn't happen, since we are Cloneable.
            throw new InternalError();
        }
    }

    /**
     * An ordinal-based serializable type-safe enumeration for component
     * alignment types as used by the {@link FormLayout}.
     */
    public static final class Alignment implements Serializable {

        private static final int HORIZONTAL = 0;
        private static final int VERTICAL = 1;
        private static final int BOTH = 2;

        private final transient String name;
        private final transient int orientation;

        private Alignment(String name, int orientation) {
            this.name = name;
            this.orientation = orientation;
        }

        static Alignment valueOf(String nameOrAbbreviation) {
            String str = nameOrAbbreviation.toLowerCase(Locale.ENGLISH);
            switch (str) {

            case "d":
            case "default":
                return DEFAULT;
            case "f":
            case "fill":
                return FILL;
            case "c":
            case "center":
                return CENTER;
            case "l":
            case "left":
                return LEFT;
            case "r":
            case "right":
                return RIGHT;
            case "t":
            case "top":
                return TOP;
            case "b":
            case "bottom":
                return BOTTOM;
            default:
                throw new IllegalArgumentException("Invalid alignment "
                        + nameOrAbbreviation
                        + ". Must be one of: left, center, right, top, bottom, "
                        + "fill, default, l, c, r, t, b, f, d.");
            }
        }

        /**
         * Returns this Alignment's name.
         *
         * @return this alignment's name.
         */
        @Override
        public String toString() {
            return name;
        }

        /**
         * Returns the first character of this Alignment's name. Used to
         * identify it in short format strings.
         *
         * @return the name's first character.
         */
        public char abbreviation() {
            return name.charAt(0);
        }

        private boolean isHorizontal() {
            return orientation != VERTICAL;
        }

        private boolean isVertical() {
            return orientation != HORIZONTAL;
        }

        // Serialization

        private static int nextOrdinal = 0;

        private final int ordinal = nextOrdinal++;

        private Object readResolve() {
            return VALUES[ordinal]; // Canonicalize
        }
    }
}
