/*
 * Copyright (c) 2002-2015 JGoodies Software GmbH. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of JGoodies Software GmbH nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.vaadin.featurepack.desktop.layouts.form;

import java.io.Serializable;

/**
 * Provides frequently used column and row specifications.
 */
public final class FormSpecs implements Serializable {

    private FormSpecs() {
        // Suppresses default constructor, prevents instantiation.
    }

    /**
     * An unmodifiable {@code ColumnSpec} that determines its width by computing
     * the maximum of all column component minimum widths.
     */
    public static final ColumnSpec MIN_COLSPEC = new ColumnSpec(
            ComponentSize.MINIMUM);

    /**
     * An unmodifiable {@code ColumnSpec} that determines its width by computing
     * the maximum of all column component preferred widths.
     */
    public static final ColumnSpec PREF_COLSPEC = new ColumnSpec(
            ComponentSize.PREFERRED);

    /**
     * An unmodifiable {@code ColumnSpec} that determines its preferred width by
     * computing the maximum of all column component preferred widths and its
     * minimum width by computing all column component minimum widths.
     * <p>
     */
    public static final ColumnSpec DEFAULT_COLSPEC = new ColumnSpec(
            ComponentSize.DEFAULT);

    /**
     * An unmodifiable {@code ColumnSpec} that has an initial width of 0 pixels
     * and that grows. Useful to describe <em>glue</em> columns that fill the
     * space between other columns.
     *
     * @see #GLUE_ROWSPEC
     */
    public static final ColumnSpec GLUE_COLSPEC = new ColumnSpec(
            ColumnSpec.DEFAULT, Sizes.pixel(0), FormSpec.DEFAULT_GROW);

    /**
     * Describes a logical horizontal gap between a label and an associated
     * component.
     */
    public static final ColumnSpec LABEL_COMPONENT_GAP_COLSPEC = ColumnSpec
            .createGap(Sizes.dluX(3));

    /**
     * Describes a logical horizontal gap between two related components.
     */
    public static final ColumnSpec RELATED_GAP_COLSPEC = ColumnSpec
            .createGap(Sizes.dluX(4));

    /**
     * Describes a logical horizontal gap between two unrelated components.
     */
    public static final ColumnSpec UNRELATED_GAP_COLSPEC = ColumnSpec
            .createGap(Sizes.dluX(7));

    /**
     * An unmodifiable {@code RowSpec} that determines its height by computing
     * the maximum of all column component minimum heights.
     */
    public static final RowSpec MIN_ROWSPEC = new RowSpec(
            ComponentSize.MINIMUM);

    /**
     * An unmodifiable {@code RowSpec} that determines its height by computing
     * the maximum of all column component preferred heights.
     */
    public static final RowSpec PREF_ROWSPEC = new RowSpec(
            ComponentSize.PREFERRED);

    /**
     * An unmodifiable {@code RowSpec} that determines its preferred height by
     * computing the maximum of all column component preferred heights and its
     * minimum height by computing all column component minimum heights.
     * <p>
     * Useful to let a column shrink from preferred height to minimum height if
     * the container space gets scarce.
     */
    public static final RowSpec DEFAULT_ROWSPEC = new RowSpec(
            ComponentSize.DEFAULT);

    /**
     * An unmodifiable {@code RowSpec} that has an initial height of 0 pixels
     * and that grows. Useful to describe glue rows that fill the space between
     * other rows.
     */
    public static final RowSpec GLUE_ROWSPEC = new RowSpec(RowSpec.DEFAULT,
            Sizes.pixel(0), FormSpec.DEFAULT_GROW);

    /**
     * Describes a logical vertical gap between a label and an associated
     * component.
     */
    public static final RowSpec LABEL_COMPONENT_GAP_ROWSPEC = RowSpec
            .createGap(Sizes.dluY(3));

    /**
     * Describes a logical vertical gap between two related components.
     */
    public static final RowSpec RELATED_GAP_ROWSPEC = RowSpec
            .createGap(Sizes.dluY(4));

    /**
     * Describes a logical vertical gap between two unrelated components.
     * <p>
     */
    public static final RowSpec UNRELATED_GAP_ROWSPEC = RowSpec
            .createGap(Sizes.dluY(7));

    /**
     * Describes a logical vertical narrow gap between two rows in the grid.
     * Useful if the vertical space is scarce or if an individual vertical gap
     * shall be small than the default line gap.
     */
    public static final RowSpec NARROW_LINE_GAP_ROWSPEC = RowSpec
            .createGap(Sizes.dluY(2));

    /**
     * Describes the logical vertical default gap between two rows in the grid.
     * A little bit larger than the narrow line gap.
     */
    public static final RowSpec LINE_GAP_ROWSPEC = RowSpec
            .createGap(Sizes.dluY(3));

    /**
     * Describes the logical vertical default gap between two paragraphs in the
     * layout grid. This gap is larger than the default line gap.
     */
    public static final RowSpec PARAGRAPH_GAP_ROWSPEC = RowSpec
            .createGap(Sizes.dluY(9));
}
