/*
 * Copyright (c) 2002-2015 JGoodies Software GmbH. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of JGoodies Software GmbH nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.vaadin.featurepack.desktop.layouts.form;

import java.io.Serializable;
import java.util.Locale;
import java.util.Objects;
import java.util.regex.Pattern;

/**
 * An abstract class that specifies columns and rows in FormLayout by their
 * default alignment, start size and resizing behavior. API users will use the
 * subclasses {@link ColumnSpec} and {@link RowSpec}.
 * <p>
 * Also implements the parser for encoded column and row specifications and
 * provides parser convenience behavior for its subclasses ColumnSpec and
 * RowSpec.
 */
public abstract class FormSpec implements Serializable {

    /**
     * By default put components in the left.
     */
    static final DefaultAlignment LEFT_ALIGN = new DefaultAlignment("left");

    /**
     * By default put components in the right.
     */
    static final DefaultAlignment RIGHT_ALIGN = new DefaultAlignment("right");

    /**
     * By default put the components in the top.
     */
    static final DefaultAlignment TOP_ALIGN = new DefaultAlignment("top");

    /**
     * By default put the components in the bottom.
     */
    static final DefaultAlignment BOTTOM_ALIGN = new DefaultAlignment("bottom");

    /**
     * By default put the components in the center.
     */
    static final DefaultAlignment CENTER_ALIGN = new DefaultAlignment("center");

    /**
     * By default fill the column or row.
     */
    static final DefaultAlignment FILL_ALIGN = new DefaultAlignment("fill");

    /**
     * An array of all enumeration values used to canonicalize deserialized
     * default alignments.
     */
    private static final DefaultAlignment[] VALUES = { LEFT_ALIGN, RIGHT_ALIGN,
            TOP_ALIGN, BOTTOM_ALIGN, CENTER_ALIGN, FILL_ALIGN };

    /**
     * Gives a column or row a fixed size.
     */
    public static final double NO_GROW = 0.0d;

    /**
     * The default resize weight.
     */
    public static final double DEFAULT_GROW = 1.0d;

    private static final Pattern TOKEN_SEPARATOR_PATTERN = Pattern.compile(":");

    /**
     * Holds the default alignment that will be used if a cell does not override
     * this default.
     */
    private DefaultAlignment defaultAlignment;

    /**
     * Holds the size that describes how to size this column or row.
     */
    private Size size;

    /**
     * Holds the resize weight; is 0 if not used.
     */
    private double resizeWeight;

    /**
     * Constructs a {@code FormSpec} for the given default alignment, size, and
     * resize weight. The resize weight must be a non-negative double; you can
     * use {@code NONE} as a convenience value for no resize.
     *
     * @param defaultAlignment
     *            the spec's default alignment
     * @param size
     *            a constant, component or bounded size
     * @param resizeWeight
     *            the spec resize weight
     *
     * @throws NullPointerException
     *             if the {@code size} is {@code null}
     * @throws IllegalArgumentException
     *             if the {@code resizeWeight} is negative
     */
    protected FormSpec(DefaultAlignment defaultAlignment, Size size,
            double resizeWeight) {
        Objects.requireNonNull(size, "Size must not be null");
        if (resizeWeight < 0) {
            throw new IllegalArgumentException(
                    "Resize weight must be non-negative");
        }

        this.defaultAlignment = defaultAlignment;
        this.size = size;
        this.resizeWeight = resizeWeight;
    }

    /**
     * Constructs a FormSpec from the specified encoded description. The
     * description will be parsed to set initial values.
     *
     * @param defaultAlignment
     *            the default alignment
     * @param encodedDescription
     *            the encoded description
     */
    protected FormSpec(DefaultAlignment defaultAlignment,
            String encodedDescription) {
        this(defaultAlignment, ComponentSize.DEFAULT, NO_GROW);
        parseAndInitValues(encodedDescription.toLowerCase(Locale.ENGLISH));
    }

    /**
     * Returns the default alignment.
     *
     * @return the default alignment
     */
    public final DefaultAlignment getDefaultAlignment() {
        return defaultAlignment;
    }

    /**
     * Returns the size.
     *
     * @return the size
     */
    public final Size getSize() {
        return size;
    }

    /**
     * Returns the current resize weight.
     *
     * @return the resize weight.
     */
    public final double getResizeWeight() {
        return resizeWeight;
    }

    /**
     * Returns if this is a horizontal specification (vs. vertical). Used to
     * distinct between horizontal and vertical dialog units, which have
     * different conversion factors.
     *
     * @return true for horizontal, false for vertical
     */
    abstract boolean isHorizontal();

    void setDefaultAlignment(DefaultAlignment defaultAlignment) {
        this.defaultAlignment = defaultAlignment;
    }

    void setSize(Size size) {
        this.size = size;
    }

    void setResizeWeight(double resizeWeight) {
        this.resizeWeight = resizeWeight;
    }

    /**
     * Parses an encoded form specification and initializes all required fields.
     * The encoded description must be in lower case.
     *
     * @param encodedDescription
     *            the FormSpec in an encoded format
     *
     * @throws NullPointerException
     *             if {@code encodedDescription} is {@code null}
     * @throws IllegalArgumentException
     *             if {@code encodedDescription} is empty, whitespace, has no
     *             size, or is otherwise invalid
     */
    private void parseAndInitValues(String encodedDescription) {
        Objects.requireNonNull(size, "Encoded description must not be null");

        String[] token = TOKEN_SEPARATOR_PATTERN.split(encodedDescription);
        if (token.length == 0) {
            throw new IllegalArgumentException(
                    "The form spec must not be empty.");
        }

        int nextIndex = 0;
        String next = token[nextIndex++];

        // Check if the first token is an orientation.
        DefaultAlignment alignment = DefaultAlignment.valueOf(next,
                isHorizontal());
        if (alignment != null) {
            setDefaultAlignment(alignment);
            if (token.length < 2) {
                throw new IllegalArgumentException(
                        "The form spec must provide a size.");
            }
            next = token[nextIndex++];
        }
        setSize(parseSize(next));
        if (nextIndex < token.length) {
            setResizeWeight(parseResizeWeight(token[nextIndex]));
        }
    }

    /**
     * Parses an encoded size spec and returns the size.
     *
     * @param token
     *            a token that represents a size, either bounded or plain
     * @return the decoded Size
     */
    private static Size parseSize(String token) {
        // TODO: add support for bounded size
        return parseAtomicSize(token);
    }

    /**
     * Decodes and returns an atomic size that is either a constant size or a
     * component size.
     *
     * @param token
     *            the encoded size
     * @return the decoded size either a constant or component size
     */
    private static Size parseAtomicSize(String token) {
        String trimmedToken = token.trim();
        ComponentSize componentSize = ComponentSize.decode(trimmedToken);
        if (componentSize != null) {
            return componentSize;
        }
        return ConstantSize.decode(trimmedToken);
    }

    /**
     * Decodes an encoded resize mode and resize weight and answers the resize
     * weight.
     *
     * @param token
     *            the encoded resize weight
     * @return the decoded resize weight
     * @throws IllegalArgumentException
     *             if the string description is an invalid string representation
     */
    private static double parseResizeWeight(String token) {
        if (token.equals("g") || token.equals("grow")) {
            return DEFAULT_GROW;
        }
        if (token.equals("n") || token.equals("nogrow")
                || token.equals("none")) {
            return NO_GROW;
        }
        // Must have format: grow(<double>)
        if ((token.startsWith("grow(") || token.startsWith("g("))
                && token.endsWith(")")) {
            int leftParen = token.indexOf('(');
            int rightParen = token.indexOf(')');
            String substring = token.substring(leftParen + 1, rightParen);
            return Double.parseDouble(substring);
        }
        throw new IllegalArgumentException("The resize argument '" + token
                + "' is invalid. "
                + " Must be one of: grow, g, none, n, grow(<double>), g(<double>)");
    }

    private static boolean isConstant(Size aSize) {
        return aSize instanceof ConstantSize;
    }

    /**
     * An ordinal-based serializable type-safe enumeration for the column and
     * row default alignment types.
     */
    public static final class DefaultAlignment implements Serializable {

        private final transient String name;

        private DefaultAlignment(String name) {
            this.name = name;
        }

        /**
         * Returns a DefaultAlignment that corresponds to the specified string,
         * null if no such alignment exists.
         *
         * @param str
         *            the encoded alignment
         * @param isHorizontal
         *            indicates the values orientation
         * @return the corresponding DefaultAlignment or null
         */
        private static DefaultAlignment valueOf(String str,
                boolean isHorizontal) {
            if (str.equals("f") || str.equals("fill")) {
                return FILL_ALIGN;
            } else if (str.equals("c") || str.equals("center")) {
                return CENTER_ALIGN;
            } else if (isHorizontal) {
                if (str.equals("r") || str.equals("right")) {
                    return RIGHT_ALIGN;
                } else if (str.equals("l") || str.equals("left")) {
                    return LEFT_ALIGN;
                } else {
                    return null;
                }
            } else {
                if (str.equals("t") || str.equals("top")) {
                    return TOP_ALIGN;
                } else if (str.equals("b") || str.equals("bottom")) {
                    return BOTTOM_ALIGN;
                } else {
                    return null;
                }
            }
        }

        /**
         * Returns this Alignment's name.
         *
         * @return this alignment's name.
         */
        @Override
        public String toString() {
            return name;
        }

        // Serialization

        private static int nextOrdinal = 0;

        private final int ordinal = nextOrdinal++;

        private Object readResolve() {
            return VALUES[ordinal]; // Canonicalize
        }
    }
}
