/*
 * Copyright (c) 2002-2015 JGoodies Software GmbH. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of JGoodies Software GmbH nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.vaadin.featurepack.desktop.layouts.form;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import com.vaadin.featurepack.desktop.layouts.AbstractLayout;
import com.vaadin.featurepack.desktop.layouts.form.CellConstraints.Alignment;
import com.vaadin.flow.component.AttachEvent;
import com.vaadin.flow.component.Component;

/**
 * {@code FormLayout} is a flexible layout that aligns components in a
 * rectangular grid of cells, with each component occupying one or more cells.
 *
 * @author Vaadin Ltd
 */
public class FormLayout extends AbstractLayout {
    /**
     * Holds the column specifications.
     *
     * @see ColumnSpec
     */
    private final List<ColumnSpec> columnSpecs;

    /**
     * Holds the row specifications.
     *
     * @see RowSpec
     */
    private final List<RowSpec> rowSpecs;

    /**
     * Maps components to their associated {@code CellConstraints}.
     *
     * @see CellConstraints
     * @see #getConstraints(Component)
     * @see #setConstraints(Component, CellConstraints)
     */
    private final Map<Component, CellConstraints> constraintMap = new HashMap<>();

    /**
     * Constructs a new {@code FormLayout}.
     */
    public FormLayout() {
        this(new ColumnSpec[0], new RowSpec[0]);
    }

    /**
     * Constructs a {@code FormLayout} using the given column and row
     * specifications.
     *
     * @param encodedColumnSpecs
     *            encoded column specifications, not {@code null}.
     * @param encodedRowSpecs
     *            encoded row specifications, not {@code null}.
     */
    public FormLayout(String encodedColumnSpecs, String encodedRowSpecs) {
        this(ColumnSpec.decodeSpecs(encodedColumnSpecs),
                RowSpec.decodeSpecs(encodedRowSpecs));
    }

    /**
     * Constructs a {@code FormLayout} using the given column and row
     * specifications.
     *
     * @param colSpecs
     *            an array of column specifications.
     * @param rowSpecs
     *            an array of row specifications.
     * @throws NullPointerException
     *             if {@code colSpecs} or {@code rowSpecs} is {@code null}
     */
    public FormLayout(ColumnSpec[] colSpecs, RowSpec[] rowSpecs) {
        Objects.requireNonNull(colSpecs, "Column specs must not be null");
        Objects.requireNonNull(rowSpecs, "Row specs must not be null");

        this.columnSpecs = new ArrayList<>(Arrays.asList(colSpecs));
        this.rowSpecs = new ArrayList<>(Arrays.asList(rowSpecs));
    }

    @Override
    protected void onContainerAttach(AttachEvent event) {
        event.getUI().getPage().addStyleSheet("vfp-form-layout.css");
    }

    @Override
    public void setContainer(Component container) {
        Component current = getContainer();

        if (current != null) {
            current.getElement().removeAttribute("vfp-form-layout");

            current.getStyle().set("--vfp-grid-columns", null);
            current.getStyle().set("--vfp-grid-rows", null);

            constraintMap.forEach((component, constraints) -> {
                clearComponent(component);
            });

            constraintMap.clear();
        }

        super.setContainer(container);

        if (container != null) {
            container.getElement().setAttribute("vfp-form-layout", "");

            getContainer().getStyle().set("--vfp-grid-columns", parseColumns());
            getContainer().getStyle().set("--vfp-grid-rows", parseRows());
        }
    }

    @Override
    public void addComponent(Component component) {
        Objects.requireNonNull(component, "Component must not be null");
    }

    @Override
    public void removeComponent(Component component) {
        Objects.requireNonNull(component, "Component must not be null");

        constraintMap.remove(component);
        clearComponent(component);
    }

    private void clearComponent(Component component) {
        component.getStyle().remove("--vfp-grid-row-start");
        component.getStyle().remove("--vfp-grid-row-end");

        component.getStyle().remove("--vfp-grid-column-start");
        component.getStyle().remove("--vfp-grid-column-end");

        component.getStyle().remove("--vfp-justify-self");
        component.getStyle().remove("--vfp-align-self");
    }

    @Override
    public void setConstraints(Component component, Object constraints) {
        Objects.requireNonNull(component, "Component must not be null");
        Objects.requireNonNull(constraints, "Constraints must not be null");

        if (constraints instanceof CellConstraints) {
            CellConstraints cc = ((CellConstraints) constraints).clone();
            ensureValidGridBounds(cc, getColumnCount(), getRowCount());
            applyConstraints(component, cc);
        } else {
            throw new IllegalArgumentException(
                    "Constraints must be a CellConstraints");
        }
    }

    /**
     * Checks and verifies that this constraints object has valid grid index
     * values, i. e. the display area cells are inside the form's grid.
     *
     * @param constraints
     *            constraints
     * @param colCount
     *            number of columns in the grid
     * @param rowCount
     *            number of rows in the grid
     * @throws IndexOutOfBoundsException
     *             if the display area described by this constraints object is
     *             not inside the grid
     */
    private void ensureValidGridBounds(CellConstraints constraints,
            int colCount, int rowCount) {
        if (constraints.gridX <= 0) {
            throw new IndexOutOfBoundsException("The column index "
                    + constraints.gridX + " must be positive.");
        }
        if (constraints.gridX > colCount) {
            throw new IndexOutOfBoundsException("The column index "
                    + constraints.gridX + " must be less than or equal to "
                    + colCount + ".");
        }
        if (constraints.gridX + constraints.gridWidth - 1 > colCount) {
            throw new IndexOutOfBoundsException("The grid width "
                    + constraints.gridWidth + " must be less than or equal to "
                    + (colCount - constraints.gridX + 1) + ".");
        }
        if (constraints.gridY <= 0) {
            throw new IndexOutOfBoundsException("The row index "
                    + constraints.gridY + " must be positive.");
        }
        if (constraints.gridY > rowCount) {
            throw new IndexOutOfBoundsException("The row index "
                    + constraints.gridY + " must be less than or equal to "
                    + rowCount + ".");
        }
        if (constraints.gridY + constraints.gridHeight - 1 > rowCount) {
            throw new IndexOutOfBoundsException("The grid height "
                    + constraints.gridHeight + " must be less than or equal to "
                    + (rowCount - constraints.gridY + 1) + ".");
        }
    }

    private void applyConstraints(Component component,
            CellConstraints constraints) {
        constraintMap.put(component, constraints);
        setCellCoordinates(component, constraints);
        setCellAlignment(component, constraints);
    }

    @Override
    public Object getConstraints(Component component) {
        Objects.requireNonNull(component, "Component must not be null");

        CellConstraints constraints = constraintMap.get(component);
        if (constraints == null) {
            throw new IllegalArgumentException(
                    "There is no constraints for the given component.");
        }

        return constraints.clone();
    }

    /**
     * Returns the number of columns in this layout.
     *
     * @return the number of columns
     */
    public int getColumnCount() {
        return columnSpecs.size();
    }

    /**
     * Returns the {@code ColumnSpec} at the specified column index.
     *
     * @param columnIndex
     *            the column index of the requested {@code ColumnSpec}
     * @return the {@code ColumnSpec} at the specified column
     * @throws IndexOutOfBoundsException
     *             if the column index is out of range
     */
    public ColumnSpec getColumnSpec(int columnIndex) {
        return columnSpecs.get(columnIndex - 1);
    }

    /**
     * Appends the given column specification to the right hand side of all
     * columns.
     *
     * @param columnSpec
     *            the column specification to be added
     * @throws NullPointerException
     *             if {@code columnSpec} is {@code null}
     */
    public void appendColumn(ColumnSpec columnSpec) {
        Objects.requireNonNull(columnSpec, "Column spec must not be null");
        columnSpecs.add(columnSpec);
        getContainer().getStyle().set("--vfp-grid-columns", parseColumns());
    }

    /**
     * Inserts the specified column at the specified position. Shifts components
     * that intersect the new column to the right hand side.
     * <p>
     * The component shift works as follows: components that were located on the
     * right hand side of the inserted column are shifted one column to the
     * right; component column span is increased by one if it intersects the new
     * column.
     *
     * @param columnIndex
     *            index of the column to be inserted
     * @param columnSpec
     *            specification of the column to be inserted
     * @throws IndexOutOfBoundsException
     *             if the column index is out of range
     */
    public void insertColumn(int columnIndex, ColumnSpec columnSpec) {
        if (columnIndex < 1 || columnIndex > getColumnCount()) {
            throw new IndexOutOfBoundsException("The column index "
                    + columnIndex + "must be in the range [1, "
                    + getColumnCount() + "].");
        }
        columnSpecs.add(columnIndex - 1, columnSpec);
        getContainer().getStyle().set("--vfp-grid-columns", parseColumns());
        shiftComponentsHorizontally(columnIndex, false);
    }

    /**
     * Removes the column with the given column index from the layout. The
     * column must not contain components.
     * <p>
     * The component shift works as follows: components that were located on the
     * right hand side of the removed column are moved one column to the left;
     * component column span is decreased by one if it intersects the removed
     * column.
     *
     * @param columnIndex
     *            index of the column to remove
     * @throws IndexOutOfBoundsException
     *             if the column index is out of range
     * @throws IllegalStateException
     *             if the column contains components
     */
    public void removeColumn(int columnIndex) {
        if (columnIndex < 1 || columnIndex > getColumnCount()) {
            throw new IndexOutOfBoundsException("The column index "
                    + columnIndex + " must be in the range [1, "
                    + getColumnCount() + "].");
        }
        columnSpecs.remove(columnIndex - 1);
        getContainer().getStyle().set("--vfp-grid-columns", parseColumns());
        shiftComponentsHorizontally(columnIndex, true);
    }

    /**
     * Returns the number of rows in this layout.
     *
     * @return the number of rows
     */
    public int getRowCount() {
        return rowSpecs.size();
    }

    /**
     * Returns the {@code RowSpec} at the specified row index.
     *
     * @param rowIndex
     *            the row index of the requested {@code RowSpec}
     * @return the {@code RowSpec} at the specified row
     * @throws IndexOutOfBoundsException
     *             if the row index is out of range
     */
    public RowSpec getRowSpec(int rowIndex) {
        return rowSpecs.get(rowIndex - 1);
    }

    /**
     * Appends the given row specification to the bottom of all rows.
     *
     * @param rowSpec
     *            the row specification to be added to the form layout
     * @throws NullPointerException
     *             if {@code rowSpec} is {@code null}
     */
    public void appendRow(RowSpec rowSpec) {
        Objects.requireNonNull(rowSpec, "Row spec must not be null");
        rowSpecs.add(rowSpec);
        getContainer().getStyle().set("--vfp-grid-rows", parseRows());
    }

    /**
     * Inserts the specified row at the specified position. Shifts components
     * that intersect the new row to the bottom and readjusts row groups.
     * <p>
     *
     * The component shift works as follows: components that were located below
     * the inserted row are shifted one row to the bottom; component row span is
     * increased by one if it intersects the new row.
     * <p>
     *
     * Row group indices that are greater or equal than the given row index will
     * be increased by one.
     *
     * @param rowIndex
     *            index of the row to be inserted
     * @param rowSpec
     *            specification of the row to be inserted
     * @throws IndexOutOfBoundsException
     *             if the row index is out of range
     */
    public void insertRow(int rowIndex, RowSpec rowSpec) {
        if (rowIndex < 1 || rowIndex > getRowCount()) {
            throw new IndexOutOfBoundsException("The row index " + rowIndex
                    + " must be in the range [1, " + getRowCount() + "].");
        }
        rowSpecs.add(rowIndex - 1, rowSpec);
        getContainer().getStyle().set("--vfp-grid-rows", parseRows());
        shiftComponentsVertically(rowIndex, false);
    }

    /**
     * Removes the row with the given row index from the layout. The row must
     * not contain components
     * <p>
     * The component shift works as follows: components that were located below
     * the removed row are moved up one row; component row span is decreased by
     * one if it intersects the removed row.
     *
     * @param rowIndex
     *            index of the row to remove
     * @throws IndexOutOfBoundsException
     *             if the row index is out of range
     * @throws IllegalStateException
     *             if the row contains components or if the row is already
     *             grouped
     */
    public void removeRow(int rowIndex) {
        if (rowIndex < 1 || rowIndex > getRowCount()) {
            throw new IndexOutOfBoundsException("The row index " + rowIndex
                    + "must be in the range [1, " + getRowCount() + "].");
        }
        rowSpecs.remove(rowIndex - 1);
        getContainer().getStyle().set("--vfp-grid-rows", parseRows());
        shiftComponentsVertically(rowIndex, true);
    }

    /**
     * Shifts components horizontally, either to the right if a column has been
     * inserted or to the left if a column has been removed.
     *
     * @param columnIndex
     *            index of the column to remove
     * @param remove
     *            true for remove, false for insert
     * @throws IllegalStateException
     *             if a removed column contains components
     */
    private void shiftComponentsHorizontally(int columnIndex, boolean remove) {
        final int offset = remove ? -1 : 1;
        constraintMap.forEach((component, constraints) -> {
            int x1 = constraints.gridX;
            int w = constraints.gridWidth;
            int x2 = x1 + w - 1;
            if (x1 == columnIndex && remove) {
                throw new IllegalStateException("The removed column "
                        + columnIndex + " must not contain components.");
            } else if (x1 >= columnIndex) {
                constraints.gridX += offset;
                setCellCoordinates(component, constraints);
            } else if (x2 >= columnIndex) {
                constraints.gridWidth += offset;
                setCellCoordinates(component, constraints);
            }
        });
    }

    /**
     * Shifts components vertically, either to the bottom if a row has been
     * inserted or to the top if a row has been removed.
     *
     * @param rowIndex
     *            index of the row to remove
     * @param remove
     *            true for remove, false for insert
     * @throws IllegalStateException
     *             if a removed row contains components
     */
    private void shiftComponentsVertically(int rowIndex, boolean remove) {
        final int offset = remove ? -1 : 1;
        constraintMap.forEach((component, constraints) -> {
            int y1 = constraints.gridY;
            int h = constraints.gridHeight;
            int y2 = y1 + h - 1;
            if (y1 == rowIndex && remove) {
                throw new IllegalStateException("The removed row " + rowIndex
                        + " must not contain components.");
            } else if (y1 >= rowIndex) {
                constraints.gridY += offset;
                setCellCoordinates(component, constraints);
            } else if (y2 >= rowIndex) {
                constraints.gridHeight += offset;
                setCellCoordinates(component, constraints);
            }
        });
    }

    private void setCellCoordinates(Component component, CellConstraints cell) {
        component.getStyle().set("--vfp-grid-row-start",
                Integer.toString(cell.gridY));
        component.getStyle().set("--vfp-grid-row-end",
                Integer.toString(cell.gridY + cell.gridHeight));

        component.getStyle().set("--vfp-grid-column-start",
                Integer.toString(cell.gridX));
        component.getStyle().set("--vfp-grid-column-end",
                Integer.toString(cell.gridX + cell.gridWidth));
    }

    private void setCellAlignment(Component component, CellConstraints cell) {
        ColumnSpec colSpec = cell.gridWidth == 1 ? getColumnSpec(cell.gridX)
                : null;
        Alignment hAlign = getEffectiveAlignment(cell.hAlign, colSpec,
                CellConstraints.FILL);
        component.getStyle().set("--vfp-justify-self",
                parseAlignX(hAlign.toString()));

        RowSpec rowSpec = cell.gridHeight == 1 ? getRowSpec(cell.gridY) : null;
        Alignment vAlign = getEffectiveAlignment(cell.vAlign, rowSpec,
                CellConstraints.CENTER);
        component.getStyle().set("--vfp-align-self",
                parseAlignY(vAlign.toString()));
    }

    /**
     * Computes and returns the effective alignment. Takes into account the cell
     * alignment and the {@code FormSpec} if applicable.
     * <p>
     * If this constraints object doesn't belong to a single column or row, the
     * {@code formSpec} parameter is {@code null}. In this case the cell
     * alignment is answered, or the fallback in case of {@code DEFAULT}.
     * <p>
     * If the cell belongs to a single column or row, we use the cell alignment,
     * unless it is {@code DEFAULT}, where the alignment is inherited from the
     * column or row.
     *
     * @param cellAlignment
     *            this cell's alignment
     * @param formSpec
     *            the associated column or row specification
     * @param fallbackAlignment
     *            the fallback alignment
     * @return the effective alignment
     */
    private Alignment getEffectiveAlignment(Alignment cellAlignment,
            FormSpec formSpec, Alignment fallbackAlignment) {
        if (formSpec == null) {
            return cellAlignment == CellConstraints.DEFAULT ? fallbackAlignment
                    : cellAlignment;
        }

        if (cellAlignment != CellConstraints.DEFAULT) {
            // Cell alignments other than DEFAULT override column /row
            // alignments
            return cellAlignment;
        }

        switch (formSpec.getDefaultAlignment().toString()) {
        case "left":
            return CellConstraints.LEFT;
        case "right":
            return CellConstraints.RIGHT;
        case "center":
            return CellConstraints.CENTER;
        case "top":
            return CellConstraints.TOP;
        case "bottom":
            return CellConstraints.BOTTOM;
        default:
            return CellConstraints.FILL;
        }
    }

    /**
     * Replaces horizontal alignment with correct CSS values. Uses null for fill
     * alignment, which is a fallback value for the custom CSS property.
     */
    private String parseAlignX(String align) {
        switch (align) {
        case "left":
            return "start";
        case "right":
            return "end";
        case "fill":
            return null;
        default:
            return align;
        }
    }

    /**
     * Replaces vertical alignment with correct CSS values. Uses null for center
     * alignment, which is a fallback value for the custom CSS property.
     */
    private String parseAlignY(String align) {
        switch (align) {
        case "top":
            return "start";
        case "bottom":
            return "end";
        case "fill":
            return "stretch";
        case "center":
            return null;
        default:
            return align;
        }
    }

    private String parseColumns() {
        return columnSpecs.stream().map(spec -> {
            if (spec.getResizeWeight() > 0) {
                return parseResizeWeight(spec.getResizeWeight());
            }

            Size size = spec.getSize();

            if (size instanceof ConstantSize) {
                ConstantSize s = (ConstantSize) size;
                if (s.getUnit() == Unit.DIALOG_UNITS) {
                    return parseHorizontalDlu(s.getValue());
                } else {
                    return s.encode();
                }
            }

            if (size instanceof ComponentSize) {
                ComponentSize s = (ComponentSize) size;
                if (s == ComponentSize.MINIMUM) {
                    return "min-content";
                } else if (s == ComponentSize.DEFAULT) {
                    return "minmax(min-content, auto)";
                }
            }

            // Used for "preferred" size.
            return "auto";
        }).collect(Collectors.joining(" "));
    }

    private String parseRows() {
        return rowSpecs.stream().map(spec -> {
            if (spec.getResizeWeight() > 0) {
                return parseResizeWeight(spec.getResizeWeight());
            }

            Size size = spec.getSize();

            if (size instanceof ConstantSize) {
                ConstantSize s = (ConstantSize) size;
                if (s.getUnit() == Unit.DIALOG_UNITS) {
                    return parseVerticalDlu(s.getValue());
                } else {
                    return s.encode();
                }
            }

            if (size instanceof ComponentSize) {
                ComponentSize s = (ComponentSize) size;
                if (s == ComponentSize.MINIMUM) {
                    return "min-content";
                } else if (s == ComponentSize.DEFAULT) {
                    return "minmax(min-content, auto)";
                }
            }

            // Used for "preferred" size.
            return "auto";
        }).collect(Collectors.joining(" "));
    }

    /**
     * Replace "resize weight" (dlu) with CSS grid fractional units so that each
     * column gets a space proportional to its weight.
     */
    private String parseResizeWeight(double weight) {
        return String.format("%dfr", (int) Math.round(weight * 100));
    }

    /**
     * Replace "dialog units" (dlu) with roughly equivalent CSS "ch" units. The
     * horizontal base unit is equal to the average width, in pixels, of the
     * characters in the system font. Each horizontal base unit is equal to 4
     * horizontal dialog units.
     */
    private String parseHorizontalDlu(double size) {
        return String.format("calc(%d / 4 * 1ch)", (int) Math.round(size));
    }

    /**
     * Replace "dialog units" (dlu) with roughly equivalent CSS "em" units. The
     * vertical base unit is equal to the height, in pixels, of the font. Each
     * vertical base unit is equal to 8 vertical dialog units.
     */
    private String parseVerticalDlu(double size) {
        return String.format("calc(%d / 8 * 1em)", (int) Math.round(size));
    }
}
