/**
 * Copyright 2000-2026 Vaadin Ltd.
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See {@literal <https://vaadin.com/commercial-license-and-service-terms>} for the full
 * license.
 */
package com.vaadin.flow.component.crud.testbench;

import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import org.openqa.selenium.By;

import com.vaadin.flow.component.button.testbench.ButtonElement;
import com.vaadin.flow.component.confirmdialog.testbench.ConfirmDialogElement;
import com.vaadin.flow.component.grid.testbench.GridElement;
import com.vaadin.flow.component.grid.testbench.GridTHTDElement;
import com.vaadin.flow.component.grid.testbench.GridTRElement;
import com.vaadin.flow.component.textfield.testbench.TextFieldElement;
import com.vaadin.testbench.ElementQuery;
import com.vaadin.testbench.TestBenchElement;
import com.vaadin.testbench.elementsbase.Element;

/**
 * TestBench element for the vaadin-crud element
 */
@Element("vaadin-crud")
public class CrudElement extends TestBenchElement {

    /**
     * Gets the crud grid
     *
     * @return the crud grid
     */
    public GridElement getGrid() {
        return this.$(GridElement.class).first();
    }

    /**
     * Gets the new item button
     *
     * @return the new item button
     */
    public Optional<TestBenchElement> getNewItemButton() {
        ElementQuery<TestBenchElement> newButtonQuery = this
                .$(TestBenchElement.class).withAttribute("slot", "new-button");
        return newButtonQuery.exists() ? Optional.of(newButtonQuery.last())
                : Optional.empty();
    }

    /**
     * Gets the filter fields if the Crud built-in Grid is being used with
     * filters enabled
     *
     * @return the filter field for each column
     */
    public List<TextFieldElement> getFilterFields() {
        return this.$(TextFieldElement.class)
                .withAttribute("crud-role", "Search").all();
    }

    /**
     * Gets the toolbar content.
     *
     * @return the toolbar content
     */
    public List<TestBenchElement> getToolbar() {
        return this.$(TestBenchElement.class).withAttribute("slot", "toolbar")
                .all();
    }

    /**
     * Opens a grid row for editing using the CRUD edit button on that row
     *
     * @param row
     *            the row to open for editing
     */
    public void openRowForEditing(int row) {
        if (isEditOnClick()) {
            this.getGrid().getCell(row, 0).click();
        } else {
            GridTRElement editedRow = getGrid().getRow(row);
            GridTHTDElement editCell = getGrid().getAllColumns().stream()
                    .map(column -> editedRow.getCell(column))
                    .filter(cell -> cell.getInnerHTML()
                            .contains("vaadin-crud-edit"))
                    .collect(Collectors.toList()).get(0);
            editCell.$("vaadin-crud-edit").get(0).click();
        }
    }

    /**
     * Gets the editor save button
     *
     * @return the editor save button
     */
    public ButtonElement getEditorSaveButton() {
        return ((TestBenchElement) getPropertyElement("_saveButton"))
                .wrap(ButtonElement.class);
    }

    /**
     * Gets the editor cancel button
     *
     * @return the editor cancel button
     */
    public ButtonElement getEditorCancelButton() {
        return ((TestBenchElement) getPropertyElement("_cancelButton"))
                .wrap(ButtonElement.class);
    }

    /**
     * Gets the editor delete button
     *
     * @return the editor delete button
     */
    public ButtonElement getEditorDeleteButton() {
        return ((TestBenchElement) getPropertyElement("_deleteButton"))
                .wrap(ButtonElement.class);
    }

    /**
     * Checks if the editor is open, either as inline editor or as a dialog
     *
     * @return {@code true} if the editor is open and {@code false}, otherwise
     */
    public boolean isEditorOpen() {
        // editorOpened can be null initially
        return Boolean.TRUE.equals(getPropertyBoolean("editorOpened"));
    }

    /**
     * Gets the editor position selected for the CRUD Possible values are ""
     * (default), "bottom" and "aside"
     *
     * @return a string containing the value defined for the editor position
     */
    public String getEditorPosition() {
        return getPropertyString("editorPosition");
    }

    /**
     * Gets whether editor can be opened by a click on the row or not
     *
     * @return {@code true} if feature is enabled or {@code false} otherwise
     */
    public boolean isEditOnClick() {
        return getPropertyBoolean("editOnClick");
    }

    /**
     * Since v25.0, returns the Crud element itself for backwards compatibility.
     *
     * @return the Crud element itself
     * @deprecated Pre v25.0, this method returned either the editor overlay,
     *             when the editor was displayed as a dialog, or the Crud
     *             itself, when the editor was displayed inline. Since v25.0,
     *             the overlay is not accessible as a separate element anymore,
     *             and, regardless whether the editor is displayed as a dialog
     *             or inline, all editor-related controls can be queried through
     *             the Crud element itself. To specifically access the editor
     *             fields, use {@link #getForm()} instead. To access the editor
     *             buttons, use {@link #getEditorSaveButton()},
     *             {@link #getEditorCancelButton()}, and
     *             {@link #getEditorDeleteButton()}.
     */
    @Deprecated(since = "25.0", forRemoval = true)
    public TestBenchElement getEditor() {
        return this;
    }

    /**
     * Gets the form element that contains form fields.
     *
     * @return the form element
     */
    public TestBenchElement getForm() {
        // Not using TestBench query here, as it would return the slot element
        // within its shadow root
        return wrapElement(findElement(By.cssSelector("[slot='form']")),
                getCommandExecutor());
    }

    /**
     * Gets the confirm cancel dialog
     *
     * @return the confirm cancel dialog
     */
    public ConfirmDialogElement getConfirmCancelDialog() {
        return $(ConfirmDialogElement.class)
                .withAttribute("slot", "confirm-cancel").first();
    }

    /**
     * Gets the confirm delete dialog
     *
     * @return the confirm delete dialog
     */
    public ConfirmDialogElement getConfirmDeleteDialog() {
        return $(ConfirmDialogElement.class)
                .withAttribute("slot", "confirm-delete").first();
    }
}
