/*
 * Copyright (C) 2000-2026 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.client.communication;

import java.util.logging.Logger;

import com.google.gwt.core.client.GWT;
import com.google.gwt.core.client.Scheduler;
import com.vaadin.client.ApplicationConfiguration;
import com.vaadin.client.ApplicationConnection;
import com.vaadin.client.ApplicationConnection.RequestStartingEvent;
import com.vaadin.client.ApplicationConnection.ResponseHandlingEndedEvent;
import com.vaadin.client.Util;
import com.vaadin.client.VLoadingIndicator;
import com.vaadin.shared.ApplicationConstants;
import com.vaadin.shared.Version;
import com.vaadin.shared.ui.ui.UIState.PushConfigurationState;

import elemental.json.Json;
import elemental.json.JsonArray;
import elemental.json.JsonObject;
import elemental.json.JsonValue;

/**
 * MessageSender is responsible for sending messages to the server.
 * <p>
 * Internally uses {@link XhrConnection} and/or {@link PushConnection} for
 * delivering messages, depending on the application configuration.
 *
 * @since 7.6
 * @author Vaadin Ltd
 */
public class MessageSender {

    private ApplicationConnection connection;
    private boolean hasActiveRequest = false;
    private boolean resynchronizeRequested = false;

    /**
     * Counter for the messages send to the server. First sent message has id 0.
     */
    private int clientToServerMessageId = 0;
    private XhrConnection xhrConnection;
    private PushConnection push;
    private JsonObject pushPendingMessage;

    public MessageSender() {
        xhrConnection = GWT.create(XhrConnection.class);
    }

    /**
     * Create a payload to use for an unload message. This can then be
     * sent via either beacon API or regular XHR.
     * 
     * @return a JSON object containing the prepared payload
     */
    private JsonObject createUnloadPayload() {
        JsonArray dummyEmptyJson = Json.createArray();
        JsonObject extraJson = Json.createObject();
        extraJson.put(ApplicationConstants.UNLOAD_BEACON, true);
        return preparePayload(dummyEmptyJson, extraJson);
    }

    /**
     * Check if the Beacon API is supported in the current browser.
     * Works as additional sanity check for odd browsers.
     * 
     * @return true if beacon API methods are present
     */
    private native boolean isBeaconAPISupported() /*-{
        if ($wnd.navigator == undefined) return false; // Sanity check
        if ($wnd.navigator.sendBeacon == undefined) return false;
        return true;
    }-*/;

    /**
     * Send an unload beacon message to the server over the XHR connection
     * to notify the server of the page having been closed.
     * 
     * @see #sendBeacon(String, String)
     */
    public void sendUnloadBeacon() {
        // Only send message through Beacon API if it's available.
        if (isBeaconAPISupported()) {
            sendBeacon(xhrConnection.getUri(), createUnloadPayload().toJson());
        } else {
            // A last effort that's unlikely to succeed as by the
            // time this method is called XHR should already be prohibited
            try {
                sendUnloadMessage();
            } catch (Exception e) {
                // Ignored, as failure is expected
            }
        }
    }

    /**
     * Send an unload message to the server over the XHR connection
     * to notify the server of the page having been closed.
     */
    public void sendUnloadMessage() {
        send(createUnloadPayload());
    }

    /**
     * Send a beacon message to the server. Beacon requests are asynchronous
     * and will be handled even after the page has closed and the rest of the
     * associated javascript is no longer active.
     * 
     * @param url target URL
     * @param payload payload (should be kept very small)
     */
    public static native void sendBeacon(String url, String payload) /*-{
        $wnd.navigator.sendBeacon(url, payload);
    }-*/;

    /**
     * Sets the application connection this instance is connected to. Called
     * internally by the framework.
     *
     * @param connection
     *            the application connection this instance is connected to
     */
    public void setConnection(ApplicationConnection connection) {
        this.connection = connection;
        xhrConnection.setConnection(connection);
    }

    private static Logger getLogger() {
        return Logger.getLogger(MessageSender.class.getName());
    }

    public void sendInvocationsToServer() {
        if (!connection.isApplicationRunning()) {
            getLogger().warning(
                    "Trying to send RPC from not yet started or stopped application");
            return;
        }

        if (hasActiveRequest() || (push != null && !push.isActive())) {
            // There is an active request or push is enabled but not active
            // -> send when current request completes or push becomes active
        } else {
            doSendInvocationsToServer();
        }
    }

    /**
     * Sends all pending method invocations (server RPC and legacy variable
     * changes) to the server.
     *
     */
    private void doSendInvocationsToServer() {
        // If there's a stored message, resend it and postpone processing the
        // rest of the queued messages to prevent resynchronization issues.
        if (pushPendingMessage != null) {
            getLogger().info("Sending pending push message "
                    + pushPendingMessage.toJson());
            JsonObject payload = pushPendingMessage;
            pushPendingMessage = null;
            startRequest();
            send(payload);
            return;
        }

        ServerRpcQueue serverRpcQueue = getServerRpcQueue();
        if (serverRpcQueue.isEmpty() && !resynchronizeRequested) {
            return;
        }

        if (ApplicationConfiguration.isDebugMode()) {
            Util.logMethodInvocations(connection, serverRpcQueue.getAll());
        }

        boolean showLoadingIndicator = serverRpcQueue.showLoadingIndicator();
        JsonArray reqJson = serverRpcQueue.toJson();
        serverRpcQueue.clear();

        if (reqJson.length() == 0 && !resynchronizeRequested) {
            // Nothing to send, all invocations were filtered out (for
            // non-existing connectors)
            getLogger().warning(
                    "All RPCs filtered out, not sending anything to the server");
            return;
        }

        JsonObject extraJson = Json.createObject();
        if (!connection.getConfiguration().isWidgetsetVersionSent()) {
            extraJson.put(ApplicationConstants.WIDGETSET_VERSION_ID,
                    Version.getFullVersion());
            connection.getConfiguration().setWidgetsetVersionSent();
        }
        if (resynchronizeRequested) {
            getLogger().info("Resynchronizing from server");
            getMessageHandler().onResynchronize();
            extraJson.put(ApplicationConstants.RESYNCHRONIZE_ID, true);
            resynchronizeRequested = false;
        }
        if (showLoadingIndicator) {
            connection.getLoadingIndicator().trigger();
        }
        send(preparePayload(reqJson, extraJson));
    }

    private ServerRpcQueue getServerRpcQueue() {
        return connection.getServerRpcQueue();
    }

    /**
     * Creates an UIDL request that can be sent to the server.
     *
     * @param reqInvocations
     *            Data containing RPC invocations and all related information.
     * @param extraJson
     *            The JsonObject whose parameters are added to the payload
     */
    protected JsonObject preparePayload(final JsonArray reqInvocations, 
                                      final JsonObject extraJson) {
        startRequest();
        JsonObject payload = Json.createObject();
        String csrfToken = getMessageHandler().getCsrfToken();
        if (!csrfToken.equals(ApplicationConstants.CSRF_TOKEN_DEFAULT_VALUE)) {
            payload.put(ApplicationConstants.CSRF_TOKEN, csrfToken);
        }
        payload.put(ApplicationConstants.RPC_INVOCATIONS, reqInvocations);
        payload.put(ApplicationConstants.SERVER_SYNC_ID,
                    getMessageHandler().getLastSeenServerSyncId());
        payload.put(ApplicationConstants.CLIENT_TO_SERVER_ID,
                    clientToServerMessageId++);
        if (extraJson != null) {
            for (String key : extraJson.keys()) {
                JsonValue value = extraJson.get(key);
                payload.put(key, value);
            }
        }
        return payload;
    }

    /**
     * Send a UIDL request to the server.
     *
     * @param reqInvocations
     *            Data containing RPC invocations and all related information.
     * @param extraJson
     *            The JsonObject whose parameters are added to the payload
     */
    protected void send(final JsonArray reqInvocations,
                        final JsonObject extraJson) {
        send(preparePayload(reqInvocations, extraJson));
    }

    /**
     * Sends an asynchronous or synchronous UIDL request to the server using the
     * given URI.
     *
     * @param payload
     *            The contents of the request to send
     */
    public void send(final JsonObject payload) {
        if (push != null && push.isBidirectional()) {
            // When using bidirectional transport, the payload is not resent
            // to the server during reconnection attempts.
            // Keep a copy of the message, so that it could be resent to the
            // server after a reconnection.
            // Reference will be cleaned up once the server confirms it has
            // seen this message
            pushPendingMessage = payload;
            push.push(payload);
        } else {
            xhrConnection.send(payload);
        }
    }

    /**
     * Sets the status for the push connection.
     *
     * @param enabled
     *            <code>true</code> to enable the push connection;
     *            <code>false</code> to disable the push connection.
     */
    public void setPushEnabled(boolean enabled) {
        final PushConfigurationState pushState = connection.getUIConnector()
                .getState().pushConfiguration;

        if (enabled && push == null) {
            push = GWT.create(PushConnection.class);
            push.init(connection, pushState);
        } else if (!enabled && push != null && push.isActive()) {
            push.disconnect(() -> {
                push = null;
                /*
                 * If push has been enabled again while we were waiting for the
                 * old connection to disconnect, now is the right time to open a
                 * new connection
                 */
                if (pushState.mode.isEnabled()) {
                    setPushEnabled(true);
                }

                /*
                 * Send anything that was enqueued while we waited for the
                 * connection to close
                 */
                if (getServerRpcQueue().isFlushPending()) {
                    getServerRpcQueue().flush();
                }
            });
        }
    }

    public void startRequest() {
        if (hasActiveRequest) {
            getLogger().severe(
                    "Trying to start a new request while another is active");
        }
        hasActiveRequest = true;
        connection.fireEvent(new RequestStartingEvent(connection));
    }

    public void endRequest() {
        if (!hasActiveRequest) {
            getLogger().severe("No active request");
        }
        // After sendInvocationsToServer() there may be a new active
        // request, so we must set hasActiveRequest to false before, not after,
        // the call.
        hasActiveRequest = false;

        if (connection.isApplicationRunning()) {
            if (getServerRpcQueue().isFlushPending()
                    || resynchronizeRequested) {
                sendInvocationsToServer();
            }
            runPostRequestHooks(connection.getConfiguration().getRootPanelId());
        }

        // deferring to avoid flickering
        Scheduler.get().scheduleDeferred(() -> {
            if (!connection.isApplicationRunning() || !(hasActiveRequest()
                    || getServerRpcQueue().isFlushPending())) {
                getLoadingIndicator().hide();

                // If on Liferay and session expiration management is in
                // use, extend session duration on each request.
                // Doing it here rather than before the request to improve
                // responsiveness.
                // Postponed until the end of the next request if other
                // requests still pending.
                extendLiferaySession();
            }
        });
        connection.fireEvent(new ResponseHandlingEndedEvent(connection));
    }

    /**
     * Runs possibly registered client side post request hooks. This is expected
     * to be run after each uidl request made by Vaadin application.
     *
     * @param appId
     */
    public static native void runPostRequestHooks(String appId)
    /*-{
        if ($wnd.vaadin.postRequestHooks) {
                for ( var hook in $wnd.vaadin.postRequestHooks) {
                        if (typeof ($wnd.vaadin.postRequestHooks[hook]) == "function") {
                                try {
                                        $wnd.vaadin.postRequestHooks[hook](appId);
                                } catch (e) {
                                }
                        }
                }
        }
    }-*/;

    /**
     * If on Liferay and logged in, ask the client side session management
     * JavaScript to extend the session duration.
     *
     * Otherwise, Liferay client side JavaScript will explicitly expire the
     * session even though the server side considers the session to be active.
     * See ticket #8305 for more information.
     */
    public static native void extendLiferaySession()
    /*-{
    if ($wnd.Liferay && $wnd.Liferay.Session) {
        $wnd.Liferay.Session.extend();
        // if the extend banner is visible, hide it
        if ($wnd.Liferay.Session.banner) {
            $wnd.Liferay.Session.banner.remove();
        }
    }
    }-*/;

    /**
     * Indicates whether or not there are currently active UIDL requests. Used
     * internally to sequence requests properly, seldom needed in Widgets.
     *
     * @return true if there are active requests
     */
    public boolean hasActiveRequest() {
        return hasActiveRequest;
    }

    /**
     * Returns a human readable string representation of the method used to
     * communicate with the server.
     *
     * @return A string representation of the current transport type
     */
    public String getCommunicationMethodName() {
        String clientToServer = "XHR";
        String serverToClient = "-";
        if (push != null) {
            serverToClient = push.getTransportType();
            if (push.isBidirectional()) {
                clientToServer = serverToClient;
            }
        }

        return "Client to server: " + clientToServer + ", "
                + "server to client: " + serverToClient;
    }

    private ConnectionStateHandler getConnectionStateHandler() {
        return connection.getConnectionStateHandler();
    }

    private MessageHandler getMessageHandler() {
        return connection.getMessageHandler();
    }

    private VLoadingIndicator getLoadingIndicator() {
        return connection.getLoadingIndicator();
    }

    /**
     * Resynchronize the client side, i.e. reload all component hierarchy and
     * state from the server
     */
    public void resynchronize() {
        getLogger().info("Resynchronize from server requested");
        resynchronizeRequested = true;
        sendInvocationsToServer();
    }

    /**
     * Used internally to update what the server expects.
     *
     * @param nextExpectedId
     *            the new client id to set
     * @param force
     *            true if the id must be updated, false otherwise
     */
    public void setClientToServerMessageId(int nextExpectedId, boolean force) {
        if (nextExpectedId == clientToServerMessageId) {
            // Everything matches they way it should
            // Remove potential pending PUSH message if it has already been seen
            // by the server.
            if (pushPendingMessage != null
                    && (int) pushPendingMessage.getNumber(
                            ApplicationConstants.CLIENT_TO_SERVER_ID) < nextExpectedId) {
                pushPendingMessage = null;
            }
            return;
        }
        if (force) {
            getLogger().info(
                    "Forced update of clientId to " + clientToServerMessageId);
            clientToServerMessageId = nextExpectedId;
            return;
        }

        if (nextExpectedId > clientToServerMessageId) {
            if (clientToServerMessageId == 0) {
                // We have never sent a message to the server, so likely the
                // server knows better (typical case is that we refreshed a
                // @PreserveOnRefresh UI)
                getLogger().info("Updating client-to-server id to "
                        + nextExpectedId + " based on server");
            } else {
                getLogger().warning(
                        "Server expects next client-to-server id to be "
                                + nextExpectedId + " but we were going to use "
                                + clientToServerMessageId + ". Will use "
                                + nextExpectedId + ".");
            }
            clientToServerMessageId = nextExpectedId;
        } else {
            // Server has not yet seen all our messages
            // Do nothing as they will arrive eventually
        }
    }

}
