/*
 * Copyright (C) 2000-2025 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.client.ui.gridlayout;

import java.util.Date;
import java.util.Map.Entry;

import com.google.gwt.core.client.Scheduler;
import com.google.gwt.core.client.Scheduler.RepeatingCommand;
import com.google.gwt.user.client.ui.Widget;
import com.vaadin.client.ComponentConnector;
import com.vaadin.client.ConnectorHierarchyChangeEvent;
import com.vaadin.client.DirectionalManagedLayout;
import com.vaadin.client.VCaption;
import com.vaadin.client.communication.StateChangeEvent;
import com.vaadin.client.ui.AbstractComponentContainerConnector;
import com.vaadin.client.ui.LayoutClickEventHandler;
import com.vaadin.client.ui.VGridLayout;
import com.vaadin.client.ui.VGridLayout.Cell;
import com.vaadin.client.ui.layout.VLayoutSlot;
import com.vaadin.shared.Connector;
import com.vaadin.shared.ui.Connect;
import com.vaadin.shared.ui.LayoutClickRpc;
import com.vaadin.shared.ui.MarginInfo;
import com.vaadin.shared.ui.gridlayout.GridLayoutServerRpc;
import com.vaadin.shared.ui.gridlayout.GridLayoutState;
import com.vaadin.shared.ui.gridlayout.GridLayoutState.ChildComponentData;
import com.vaadin.ui.GridLayout;

/**
 * A connector class for the GridLayout component.
 *
 * @author Vaadin Ltd
 */
@Connect(GridLayout.class)
public class GridLayoutConnector extends AbstractComponentContainerConnector
        implements DirectionalManagedLayout {

    private static boolean fontLoadingCallbackRegistered = false;

    private LayoutClickEventHandler clickEventHandler = new LayoutClickEventHandler(
            this) {

        @SuppressWarnings("deprecation")
        @Override
        protected ComponentConnector getChildComponent(
                com.google.gwt.user.client.Element element) {
            return getWidget().getComponent(element);
        }

        @Override
        protected LayoutClickRpc getLayoutClickRPC() {
            return getRpcProxy(GridLayoutServerRpc.class);
        }

    };

    @Override
    public void init() {
        super.init();
        getWidget().client = getConnection();

        getLayoutManager().registerDependency(this,
                getWidget().spacingMeasureElement);

        if (!fontLoadingCallbackRegistered) {
            fontLoadingCallbackRegistered = true;
            if (!registerFontLoadedCallback()) {
                /*
                 * Sometimes the initial rendering fails to use correct styles
                 * even if fonts have already loaded, add a little bit extra
                 * delay and re-measure contents.
                 *
                 * Note: this only actually triggers layouting a few times,
                 * because setNeedsMeasureRecursively only marks the component
                 * hierarchy for measurement. The layouting itself is only
                 * triggered every ~100 ms.
                 *
                 * We use a fixed delay here, because while repeated
                 * scheduleDeferred and scheduleFinally would likewise only
                 * trigger layouting a few times, and do it ASAP, it would add
                 * unnecessary overhead to traverse through the component
                 * hierarchy potentially hundreds of times to earmark all the
                 * children for layouting too.
                 */
                Scheduler.get().scheduleFixedDelay(new RepeatingCommand() {
                    Long initialTime = null;

                    @Override
                    public boolean execute() {
                        if (initialTime == null) {
                            // set at the first execution, otherwise the initial
                            // triggering can eat a very large part of the
                            // allocated 100 ms already
                            initialTime = new Date().getTime();
                        }
                        getLayoutManager().setNeedsMeasureRecursively(
                                GridLayoutConnector.this);
                        // trigger again after the same delay, unless
                        // 100 ms have passed from first execution
                        if (new Date().getTime() - initialTime < 500) {
                            // re-trigger
                            return true;
                        }
                        // cancel
                        return false;
                    }
                }, 15); // short interval for smoother experience
            }
        }
    }

    private static native boolean registerFontLoadedCallback()
    /*-{
        try {
            if (!$doc.fonts) {
                // perform delayed forced refresh
                setTimeout(function() {
                    $wnd.vaadin.forceLayout();
                }, 300);
            } else if ($doc.fonts.status == 'loading') {
                $doc.fonts.ready.then(function () {
                    $wnd.vaadin.forceLayout();
                });
            } else {
                // indicate to the calling method that a smaller
                // re-measurement could be needed
                return false;
            }
        } catch(err) {
            // just rely on the normal rendering being close to correct
        }
        return true;
    }-*/;

    @Override
    public void onUnregister() {
        VGridLayout layout = getWidget();
        getLayoutManager().unregisterDependency(this,
                layout.spacingMeasureElement);

        // Unregister caption size dependencies
        for (ComponentConnector child : getChildComponents()) {
            Cell cell = layout.widgetToCell.get(child.getWidget());
            cell.slot.setCaption(null);
        }
    }

    @Override
    public GridLayoutState getState() {
        return (GridLayoutState) super.getState();
    }

    @SuppressWarnings("deprecation")
    @Override
    public void onStateChanged(StateChangeEvent stateChangeEvent) {
        super.onStateChanged(stateChangeEvent);

        clickEventHandler.handleEventHandlerRegistration();

        getWidget().hideEmptyRowsAndColumns = getState().hideEmptyRowsAndColumns;

        initSize();

        for (Entry<Connector, ChildComponentData> entry : getState().childData
                .entrySet()) {
            ComponentConnector child = (ComponentConnector) entry.getKey();

            Cell cell = getCell(child);

            ChildComponentData childComponentData = entry.getValue();
            cell.updateCell(childComponentData);
        }

        VGridLayout layout = getWidget();
        layout.colExpandRatioArray = getState().colExpand;
        layout.rowExpandRatioArray = getState().rowExpand;

        layout.updateMarginStyleNames(
                new MarginInfo(getState().marginsBitmask));
        layout.updateSpacingStyleName(getState().spacing);
        getLayoutManager().setNeedsLayout(this);
    }

    private Cell getCell(ComponentConnector child) {
        VGridLayout layout = getWidget();
        Cell cell = layout.widgetToCell.get(child.getWidget());

        if (cell == null) {
            ChildComponentData childComponentData = getState().childData
                    .get(child);
            int row = childComponentData.row1;
            int col = childComponentData.column1;

            cell = layout.createNewCell(row, col);
        }
        return cell;
    }

    @Override
    public void onConnectorHierarchyChange(
            ConnectorHierarchyChangeEvent event) {
        VGridLayout layout = getWidget();

        // clean non rendered components
        for (ComponentConnector oldChild : event.getOldChildren()) {
            if (oldChild.getParent() == this) {
                continue;
            }

            Widget childWidget = oldChild.getWidget();
            layout.remove(childWidget);
        }

        initSize();

        for (ComponentConnector componentConnector : getChildComponents()) {
            Cell cell = getCell(componentConnector);

            cell.setComponent(componentConnector, getChildComponents());
        }
    }

    private void initSize() {
        VGridLayout layout = getWidget();
        int cols = getState().columns;
        int rows = getState().rows;

        layout.columnWidths = new int[cols];
        layout.rowHeights = new int[rows];
        layout.explicitRowRatios = getState().explicitRowRatios;
        layout.explicitColRatios = getState().explicitColRatios;
        layout.setSize(rows, cols);
    }

    @Override
    public void updateCaption(ComponentConnector childConnector) {
        VGridLayout layout = getWidget();
        Cell cell = layout.widgetToCell.get(childConnector.getWidget());
        if (VCaption.isNeeded(childConnector)) {
            VLayoutSlot layoutSlot = cell.slot;
            VCaption caption = layoutSlot.getCaption();
            if (caption == null) {
                caption = new VCaption(childConnector, getConnection());
                Widget widget = childConnector.getWidget();

                layout.setCaption(widget, caption);
            }
            caption.updateCaption();
        } else {
            layout.setCaption(childConnector.getWidget(), null);
            getLayoutManager().setNeedsLayout(this);
        }
    }

    @Override
    public VGridLayout getWidget() {
        return (VGridLayout) super.getWidget();
    }

    @Override
    public void layoutVertically() {
        getWidget().updateHeight();
    }

    @Override
    public void layoutHorizontally() {
        getWidget().updateWidth();
    }

    @Override
    protected void updateWidgetSize(String newWidth, String newHeight) {
        // Prevent the element from momentarily shrinking to zero size
        // when the size is set to undefined by a state change but before
        // it is recomputed in the layout phase. This may affect scroll
        // position in some cases; see #13386.
        if (!isUndefinedHeight()) {
            getWidget().setHeight(newHeight);
        }
        if (!isUndefinedWidth()) {
            getWidget().setWidth(newWidth);
        }
    }
}
