/*
 * Vaadin Framework 7
 *
 * Copyright (C) 2000-2025 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */

package com.vaadin.client;

import java.util.logging.Logger;

import com.google.gwt.core.client.JsArrayString;
import com.google.gwt.user.client.Command;
import com.vaadin.client.ResourceLoader.ResourceLoadEvent;
import com.vaadin.client.ResourceLoader.ResourceLoadListener;

/**
 * Handles loading of dependencies (style sheets and scripts) in the
 * application.
 *
 * Use {@link ApplicationConfiguration#runWhenDependenciesLoaded(Command)} to
 * execute a command after all dependencies have finished loading.
 *
 * @author Vaadin Ltd
 * @since 7.7.49
 */
public class DependencyLoader {

    private static final String STYLE_DEPENDENCIES = "styleDependencies";
    private static final String SCRIPT_DEPENDENCIES = "scriptDependencies";
    private ApplicationConnection connection = null;

    /**
     * Sets the ApplicationConnection this instance is connected to.
     *
     * Only used internally.
     *
     * @param connection
     *            The ApplicationConnection for this instance
     */
    public void setConnection(ApplicationConnection connection) {
        if (this.connection != null) {
            throw new IllegalStateException(
                    "Application connection has already been set");
        }
        if (connection == null) {
            throw new IllegalArgumentException(
                    "ApplicationConnection can not be null");
        }
        this.connection = connection;
    }

    /**
     * Loads the any dependencies present in the given json snippet.
     *
     * Scans the key "{@literal scriptDependencies}" for JavaScripts and the key
     * "{@literal styleDependencies}" for style sheets.
     *
     * Ensures that the given JavaScript dependencies are loaded in the given
     * order. Does not ensure anything about stylesheet order.
     *
     * @param json
     *            the JSON containing the dependencies to load
     */
    public void loadDependencies(ValueMap json) {
        if (json.containsKey(SCRIPT_DEPENDENCIES)) {
            loadScriptDependencies(json.getJSStringArray(SCRIPT_DEPENDENCIES));
        }
        if (json.containsKey(STYLE_DEPENDENCIES)) {
            loadStyleDependencies(json.getJSStringArray(STYLE_DEPENDENCIES));
        }

    }

    private void loadStyleDependencies(JsArrayString dependencies) {
        // Assuming no reason to interpret in a defined order
        ResourceLoadListener resourceLoadListener = new ResourceLoadListener() {
            @Override
            public void onLoad(ResourceLoadEvent event) {
                ApplicationConfiguration.endDependencyLoading();
            }

            @Override
            public void onError(ResourceLoadEvent event) {
                getLogger().severe(event.getResourceUrl()
                        + " could not be loaded, or the load detection failed because the stylesheet is empty.");
                // The show must go on
                onLoad(event);
            }
        };
        ResourceLoader loader = ResourceLoader.get();
        for (int i = 0; i < dependencies.length(); i++) {
            String url = translateVaadinUri(dependencies.get(i));
            ApplicationConfiguration.startDependencyLoading();
            loader.loadStylesheet(url, resourceLoadListener);
        }
    }

    private void loadScriptDependencies(final JsArrayString dependencies) {
        if (dependencies.length() == 0) {
            return;
        }

        // Listener that loads the next when one is completed
        ResourceLoadListener resourceLoadListener = new ResourceLoadListener() {
            @Override
            public void onLoad(ResourceLoadEvent event) {
                // Call start for next before calling end for current
                ApplicationConfiguration.endDependencyLoading();
            }

            @Override
            public void onError(ResourceLoadEvent event) {
                getLogger().severe(
                        event.getResourceUrl() + " could not be loaded.");
                // The show must go on
                onLoad(event);
            }
        };

        ResourceLoader loader = ResourceLoader.get();
        for (int i = 0; i < dependencies.length(); i++) {
            ApplicationConfiguration.startDependencyLoading();
            String preloadUrl = translateVaadinUri(dependencies.get(i));
            loader.loadScript(preloadUrl, resourceLoadListener);
        }
    }

    private String translateVaadinUri(String url) {
        return connection.translateVaadinUri(url);
    }

    private static Logger getLogger() {
        return Logger.getLogger(DependencyLoader.class.getName());
    }

}
