/*
 * Copyright (c) 2000-2022 Vaadin Ltd.
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
 */

// do some setup for the grid layout
(function () {
    if (!window.Vaadin.ClassicComponents) {
        window.Vaadin.ClassicComponents = {};
    }

    window.Vaadin.ClassicComponents.gridLayoutConnector = {
        initLazy: gridLayout => {
            // Check whether the connector was already initialized for the grid layout
            if (gridLayout.$connector) {
                return;
            }

            gridLayout.$connector = {
                hideEmptyRowsAndColumns: false,
                horizontalSpacing: 12, // px
                verticalSpacing: 12, // px
                resizeObserver: new ResizeObserver(() => {
                    if (!gridLayout.$connector._resizeObserverPaused) {
                        gridLayout.$connector.update();
                    }
                }),
                mutationObserver: new MutationObserver((mutations) => {
                    mutations.forEach(mutation => {
                        const { addedNodes, removedNodes } = mutation;
                        addedNodes.forEach(node => {
                            if (node.firstChild) {
                                gridLayout.$connector.resizeObserver.observe(node.firstChild);
                            }
                        });
                        removedNodes.forEach(node => {
                            if (node.firstChild) {
                                gridLayout.$connector.resizeObserver.unobserve(node.firstChild)
                            }
                        });
                    });
                }),
            };

            gridLayout.$connector.pauseResizeObserver = () => {
                gridLayout.$connector._resizeObserverPaused = true;
            };

            gridLayout.$connector.resumeResizeObserver = () => {
                gridLayout.$connector._resizeObserverPaused = false;
            };

            /**
             * Checks if the grid-layout should add spacing between cells.
             *
             * @returns {boolean} <code>true</code> if spacing should be added
             * between cells, <code>false</code> otherwise.
             */
            gridLayout.$connector.isSpacing = () => {
                return gridLayout.hasAttribute('spacing');
            };

            /**
             * Get the amount of vertical spacing between cells.
             *
             * @returns {number} the amount of vertical spacing between each cell, in pixels.
             */
            gridLayout.$connector.getVerticalSpacing = () => {
                return gridLayout.$connector.isSpacing() ? gridLayout.$connector.verticalSpacing : 0;
            };

            /**
             * Get the amount of horizontal spacing between cells.
             *
             * @returns {number} the amount of horizontal spacing between each cell, in pixels.
             */
            gridLayout.$connector.getHorizontalSpacing = () => {
                return gridLayout.$connector.isSpacing() ? gridLayout.$connector.horizontalSpacing : 0;
            };

            gridLayout.$connector.isUndefinedWidth = () => {
                return !gridLayout.classList.contains('v-has-width');
            };

            gridLayout.$connector.isUndefinedHeight = () => {
                return !gridLayout.classList.contains('v-has-height');
            };

            gridLayout.$connector.isRelativeHeight = () => {
                return gridLayout.style.height && gridLayout.style.height.endsWith('%');
            };

            gridLayout.$connector.computeCurrentGridStyles = () => {
                const gridComputedStyle = getComputedStyle(gridLayout);
                return {
                    paddingLeft: parseFloat(gridComputedStyle.paddingLeft),
                    paddingTop: parseFloat(gridComputedStyle.paddingTop),
                    paddingRight: parseFloat(gridComputedStyle.paddingRight),
                    paddingBottom: parseFloat(gridComputedStyle.paddingBottom),
                    borderRight: parseFloat(gridComputedStyle.borderRight),
                    borderBottom: parseFloat(gridComputedStyle.borderBottom),
                    marginRight: parseFloat(gridComputedStyle.marginRight),
                    marginBottom: parseFloat(gridComputedStyle.marginBottom)
                };
            }


            gridLayout.$connector.update = () => {

                // use inline-block to measure widget sizes for layouting
                gridLayout.$connector.setInlineBlockToAllWidgets();

                // Pause observer to prevent recursive calls during updates
                gridLayout.$connector.pauseResizeObserver();


                // Layout computation phase. Only query DOM element styles in this phase.
                // Do not set any DOM element styles here, otherwise you'll force
                // browser Style Recalculation or even trigger Layout. That would
                // harm performance a lot
                gridLayout.$connector.updateColumnWidths();
                gridLayout.$connector.updateRowHeights();
                gridLayout.$connector.expandRows();

                gridLayout.$connector.expandColumns();
                gridLayout.$connector.calculateUsedWidths();
                gridLayout.$connector.calculateUsedHeights();
                const gridStyles = gridLayout.$connector.computeCurrentGridStyles();

                // Layout update phase. Only set DOM element style values here, do not
                // call getComputedStyle for any DOM element in this phase or you'll force
                // browser Style Recalculation or even trigger Page Layout. That would
                // harm performance a lot.
                gridLayout.$connector.restoreOriginalDisplayStyles();
                gridLayout.$connector.clearAllCellsPaddings();
                gridLayout.$connector.layoutCellsHorizontally(gridStyles);
                gridLayout.$connector.layoutCellsVertically(gridStyles);

                // Resume observer after all operations are complete,
                // note that this has to be done in an extra task (setTimeout), so the
                // resume is executed *after* the task where resizeObserverCallback is called
                setTimeout(() => {
                    gridLayout.$connector.resumeResizeObserver();
                }, 0);
            };

            gridLayout.$connector.clearAllCellsPaddings = () => {
                for (const cell of gridLayout.children) {
                    cell.style.removeProperty('padding-left');
                    cell.style.removeProperty('padding-top');
                }
            };

            gridLayout.$connector.setInlineBlockToAllWidgets = () => {
                for (const cell of gridLayout.children) {
                    const widget = cell.firstChild;
                    // save the original value so we can restore it later
                    widget._gridLayoutOriginalDisplayValue = widget.style.display;
                    // some display values will yield e.g. an offsetHeight of 0
                    // so we need to calculate dimensions using a fix display value
                    widget.style.display = 'inline-block';
                }
            }

            gridLayout.$connector.restoreOriginalDisplayStyles = () => {
                for (const cell of gridLayout.children) {
                    const widget = cell.firstChild;
                    widget.style.display = widget._gridLayoutOriginalDisplayValue
                }
            }

            // columns

            gridLayout.$connector.updateColumnWidths = () => {
                const { columns } = gridLayout;
                gridLayout.$connector.columnWidths = Array.from({ length: columns }, () => 0);
                for (let cell of gridLayout.children) {
                    // check for colspan
                    const col = cell.x1;
                    const colspan = 1 + cell.x2 - cell.x1;
                    if (colspan === 1) {
                        const widget = cell.firstChild;
                        const cellWidth = gridLayout.$connector.getCellWidth(cell);
                        const cellHasRelativeWidth = widget.style.width && widget.style.width.endsWith('%');
                        if (!cellHasRelativeWidth && gridLayout.$connector.columnWidths[col] < cellWidth) {
                            gridLayout.$connector.columnWidths[col] = cellWidth;
                        }
                    }
                }
            };

            gridLayout.$connector.expandColumns = () => {
                const conn = gridLayout.$connector;
                if (conn.isUndefinedWidth()) {
                    return;
                }
                const minColWidths = Array.from(conn.columnWidths);
                const usedSpace = conn.calcColUsedSpace();
                const actualExpandRatio = conn.calcColumnExpandRatio();
                const gridStyles = getComputedStyle(gridLayout);
                const gridWidth = parseFloat(gridStyles.width);
                const gridPaddings = parseFloat(gridStyles.paddingLeft) + parseFloat(gridStyles.paddingRight);
                const gridBorders = parseFloat(gridStyles.borderLeft) + parseFloat(gridStyles.borderRight);
                const gridMargins = parseFloat(gridStyles.marginLeft) + parseFloat(gridStyles.marginRight);
                const availableSpace = gridWidth - gridPaddings - gridBorders - gridMargins;
                let excessSpace = availableSpace - usedSpace;
                let distributed = 0;
                if (excessSpace <= 0) {
                    return;
                }
                let expandRatioSum = actualExpandRatio.reduce((prev, curr) => prev + curr, 0);
                if (expandRatioSum === 0) {
                    // no column expanded
                    for (let i = 0; i < actualExpandRatio.length; i++) {
                        actualExpandRatio[i] = 1;
                    }
                    expandRatioSum = actualExpandRatio.length;
                }
                for (let i = 0; i < conn.columnWidths.length; i++) {
                    const ew = excessSpace * actualExpandRatio[i] / expandRatioSum;
                    conn.columnWidths[i] = minColWidths[i] + ew;
                    distributed += ew;
                }
                excessSpace -= distributed;
                let counter = 0;
                while (excessSpace > 0) {
                    conn.columnWidths[counter % conn.columnWidths.length]++;
                    excessSpace--;
                    counter++;
                }
            };

            gridLayout.$connector.calcColUsedSpace = () => {
                const conn = gridLayout.$connector;
                let usedSpace = 0;
                for (let i = 0; i < conn.columnWidths.length; i++) {
                    if (conn.columnWidths[i] > 0 || !conn.hiddenEmptyColumn(i)) {
                        const isFirstVisibleColumn = usedSpace === 0;
                        usedSpace += conn.columnWidths[i];
                        if (!isFirstVisibleColumn) {
                            usedSpace += conn.getHorizontalSpacing();
                        }
                    }
                }
                return usedSpace;
            };

            gridLayout.$connector.calcColumnExpandRatio = () => {
                const actualExpandRatio = [];
                const conn = gridLayout.$connector;
                for (let i = 0; i < conn.columnWidths.length; i++) {
                    if (conn.hiddenEmptyColumn(i)) {
                        actualExpandRatio.push(0);
                    } else {
                        actualExpandRatio.push(gridLayout.colExpandRatios[i]);
                    }
                }
                return actualExpandRatio;
            };

            gridLayout.$connector.hiddenEmptyColumn = (columnIndex) => {
                const conn = gridLayout.$connector;
                return conn.hideEmptyRowsAndColumns
                    && !conn.colHasComponentsOrColSpan(columnIndex)
                    && gridLayout.colExpandRatios[columnIndex] === 0;
            };

            gridLayout.$connector.colHasComponentsOrColSpan = (columnIndex) => {
                const cells = Array.from(gridLayout.children);
                return cells.some(cell => cell.x1 <= columnIndex && cell.x2 <= columnIndex);
            };

            gridLayout.$connector.calculateUsedWidths = () => {
                for (const cell of gridLayout.children) {
                    const widget = cell.firstChild;
                    const widgetStyles = getComputedStyle(widget);
                    const widgetMargins = parseFloat(widgetStyles.marginLeft) + parseFloat(widgetStyles.marginRight);
                    widget._gridLayoutUsedWidth = widget.offsetWidth + widgetMargins;
                }
            }

            gridLayout.$connector.layoutCellsHorizontally = (gridComputedStyle) => {
                let left = gridComputedStyle.paddingLeft;
                const cells = Array.from(gridLayout.children);
                const { rows, columns: cols } = gridLayout;
                for (let col = 0; col < cols; col++) {
                    for (let row = 0; row < rows; row++) {
                        const cell = cells.find(c => c.x1 === col && c.y1 === row);
                        if (cell) {
                            gridLayout.$connector.layoutCellHorizontally(cell, left, gridComputedStyle.paddingLeft);
                        }
                    }
                    if (!gridLayout.$connector.hideEmptyRowsAndColumns
                        || gridLayout.$connector.colHasComponentsOrColSpan(col)
                        || gridLayout.$connector.columnWidths[col] > 0) {
                        if (left > gridComputedStyle.paddingLeft) {
                            left += gridLayout.$connector.getHorizontalSpacing();
                        }
                        left += gridLayout.$connector.columnWidths[col];
                    }
                }
                if (gridLayout.$connector.isUndefinedWidth()) {
                    const paddingRight = gridComputedStyle.paddingRight;
                    const borderRight = gridComputedStyle.borderRight;
                    const marginRight = gridComputedStyle.marginRight;
                    gridLayout.style.width = left + paddingRight + borderRight + marginRight + 'px';
                }
            };

            // rows

            gridLayout.$connector.updateRowHeights = () => {
                const { rows } = gridLayout;
                gridLayout.$connector.rowHeights = Array.from({ length: rows }, () => 0);
                for (let cell of gridLayout.children) {
                    // check for rowspan
                    const row = cell.y1;
                    const rowspan = 1 + cell.y2 - cell.y1;
                    const widget = cell.firstChild;
                    const widgetStyles = getComputedStyle(widget);
                    const widgetPaddings = parseFloat(widgetStyles.paddingTop) + parseFloat(widgetStyles.paddingBottom);
                    const widgetMargins = parseFloat(widgetStyles.marginTop) + parseFloat(widgetStyles.marginBottom);
                    var cellHeight = widget.offsetHeight + widgetPaddings + widgetMargins;

                    // TextField is special: offsetHeight is much larger for it than expected.
                    // Artificially deduct the delta to look it nicer.
                    if (widget.nodeName === 'VAADIN-TEXT-FIELD' && widget.offsetHeight > 25) {
                        cellHeight = 60 + widgetPaddings + widgetMargins;
                        if (widget.hasAttribute('has-helper') && !gridLayout.$connector.isSpacing() && cellHeight < 100) {
                            cellHeight = cellHeight + 20;
                        }
                    }

                    const cellHasRelativeHeight = cell.style.height && cell.style.height.endsWith('%');
                    if (!cellHasRelativeHeight && gridLayout.$connector.rowHeights[row] < cellHeight) {
                        cellHeight = (rowspan === 1) ? cellHeight : (cellHeight / rowspan);
                        gridLayout.$connector.rowHeights[row] = cellHeight;
                    }
                }
            };

            gridLayout.$connector.expandRows = () => {
                const conn = gridLayout.$connector;
                if (conn.isUndefinedHeight()) {
                    return;
                }
                const minRowHeights = Array.from(conn.rowHeights);
                const usedSpace = conn.calcRowUsedSpace();
                const actualExpandRatio = conn.calcRowExpandRatio();
                const gridStyles = getComputedStyle(gridLayout);
                const gridHeight = parseFloat(gridStyles.height);
                const gridPaddings = parseFloat(gridStyles.paddingTop) + parseFloat(gridStyles.paddingBottom);
                const gridBorders = parseFloat(gridStyles.borderTop) + parseFloat(gridStyles.borderBottom);
                const gridMargins = parseFloat(gridStyles.marginTop) + parseFloat(gridStyles.marginBottom);
                const availableSpace = gridHeight - gridPaddings - gridBorders - gridMargins;
                let excessSpace = availableSpace - usedSpace;
                let distributed = 0;
                if (excessSpace <= 0) {
                    return;
                }
                let expandRatioSum = 0;
                for (let i = 0; i < conn.rowHeights.length; i++) {
                    expandRatioSum += actualExpandRatio[i];
                }
                if (expandRatioSum === 0) {
                    // no rows are expanded
                    for (let i = 0; i < actualExpandRatio.length; i++) {
                        actualExpandRatio[i] = 1;
                    }
                    expandRatioSum = actualExpandRatio.length;
                }
                for (let i = 0; i < conn.rowHeights.length; i++) {
                    const ew = excessSpace * actualExpandRatio[i] / expandRatioSum;
                    conn.rowHeights[i] = minRowHeights[i] + ew;
                    distributed += ew;
                }
                excessSpace -= distributed;
                let counter = 0;
                while (excessSpace > 0) {
                    conn.rowHeights[counter % conn.rowHeights.length]++;
                    excessSpace--;
                    counter++;
                }
            };

            gridLayout.$connector.calcRowUsedSpace = () => {
                const conn = gridLayout.$connector;
                let usedSpace = 0;
                for (let i = 0; i < conn.rowHeights.length; i++) {
                    if (conn.rowHeights[i] > 0 || !conn.hiddenEmptyRow(i)) {
                        const isFirstVisibleRow = usedSpace === 0;
                        usedSpace += conn.rowHeights[i];
                        if (!isFirstVisibleRow) {
                            usedSpace += conn.getVerticalSpacing();
                        }
                    }
                }
                return usedSpace;
            };

            gridLayout.$connector.calcRowExpandRatio = () => {
                const actualExpandRatio = [];
                const conn = gridLayout.$connector;
                for (let i = 0; i < conn.rowHeights.length; i++) {
                    if (conn.hiddenEmptyRow(i)) {
                        actualExpandRatio.push(0);
                    } else {
                        actualExpandRatio.push(gridLayout.rowExpandRatios[i]);
                    }
                }
                return actualExpandRatio;
            };

            gridLayout.$connector.hiddenEmptyRow = (rowIndex) => {
                const conn = gridLayout.$connector;
                return conn.hideEmptyRowsAndColumns
                    && !conn.rowHasComponentsOrRowSpan(rowIndex)
                    && gridLayout.rowExpandRatios[rowIndex] === 0;
            };

            gridLayout.$connector.rowHasComponentsOrRowSpan = (rowIndex) => {
                const cells = Array.from(gridLayout.children);
                return cells.some(cell => cell.y1 <= rowIndex && cell.y2 <= rowIndex);
            };

            gridLayout.$connector.calculateUsedHeights = () => {
                for (const cell of gridLayout.children) {
                    const widget = cell.firstChild;
                    const widgetStyles = getComputedStyle(widget);
                    const widgetMargins = parseFloat(widgetStyles.marginTop) + parseFloat(widgetStyles.marginBottom);
                    widget._gridLayoutUsedHeight = widget.offsetHeight + widgetMargins;
                }
            }

            gridLayout.$connector.layoutCellsVertically = (gridComputedStyle) => {
                const cells = Array.from(gridLayout.children);
                const { rows, columns: cols } = gridLayout;
                const gridPaddingTop = gridComputedStyle.paddingTop;
                let top = gridPaddingTop;
                for (let col = 0; col < cols; col++) {
                    top = gridPaddingTop;
                    for (let row = 0; row < rows; row++) {
                        const cell = cells.find(c => c.x1 === col && c.y1 === row);
                        if (cell) {
                            gridLayout.$connector.layoutCellVertically(cell, top, gridPaddingTop);
                        }
                        if (!gridLayout.$connector.hideEmptyRowsAndColumns
                            || gridLayout.$connector.rowHasComponentsOrRowSpan(row)
                            || gridLayout.$connector.rowHeights[row] > 0) {
                            if (top > gridPaddingTop) {
                                top += gridLayout.$connector.getVerticalSpacing();
                            }
                            top += gridLayout.$connector.rowHeights[row];
                        }
                    }
                }
                if (gridLayout.$connector.isUndefinedHeight() || gridLayout.$connector.isRelativeHeight()) {
                    const paddingBottom = gridComputedStyle.paddingBottom;
                    const borderBottom = gridComputedStyle.borderBottom;
                    const marginBottom = gridComputedStyle.marginBottom;
                    gridLayout.style.height = top + paddingBottom + borderBottom + marginBottom + 'px';
                }
            };

            // cells

            gridLayout.$connector.layoutCellHorizontally = (cell, left, gridPaddingLeft) => {
                const availableWidth = gridLayout.$connector.getAvailableWidth(cell);
                cell.style.width = availableWidth + 'px';
                cell.style.left = left + 'px';
                if (left > gridPaddingLeft) {
                    cell.style.paddingLeft = gridLayout.$connector.getHorizontalSpacing() + 'px';
                }

                // handle content alignment
                const widget = cell.firstChild;
                if (cell.horizontalAlignment && cell.horizontalAlignment !== 'left') {
                    const usedWidth = widget._gridLayoutUsedWidth;
                    let leftPadding = availableWidth - usedWidth;
                    if (cell.horizontalAlignment === 'center') {
                        leftPadding /= 2;
                    }
                    if (left > gridPaddingLeft) {
                        leftPadding += gridLayout.$connector.getHorizontalSpacing();
                    }
                    const roundedPadding = Math.round(leftPadding);
                    widget.style.left = roundedPadding + 'px';
                } else {
                    widget.style.left = '';
                }
            };

            gridLayout.$connector.layoutCellVertically = (cell, top, gridPaddingTop) => {
                const cellHeight = gridLayout.$connector.getAvailableHeight(cell);
                cell.style.height = cellHeight + 'px';
                cell.style.top = top + 'px';
                if (top > gridPaddingTop) {
                    cell.style.paddingTop = gridLayout.$connector.getVerticalSpacing() + 'px';
                }

                // handle content alignment
                const widget = cell.firstChild;
                if (cell.verticalAlignment && cell.verticalAlignment !== 'top') {
                    const usedHeight = widget._gridLayoutUsedHeight;
                    let topPadding;
                    if (cell.verticalAlignment === 'middle') {
                        topPadding = (cellHeight - usedHeight) / 2;
                    } else {
                        topPadding = (cellHeight - usedHeight);
                    }
                    if (top > 0) {
                        topPadding += gridLayout.$connector.getVerticalSpacing();
                    }
                    widget.style.top = topPadding + 'px';
                } else {
                    widget.style.top = '';
                }
            };

            gridLayout.$connector.getCellWidth = (cell) => {
                const widget = cell.firstChild;
                const widgetStyles = getComputedStyle(widget);
                const widgetMargins = parseFloat(widgetStyles.marginLeft) + parseFloat(widgetStyles.marginRight);
                const widgetPaddings = parseFloat(widgetStyles.paddingLeft) + parseFloat(widgetStyles.paddingRight);
                const cellWidth = widget.offsetWidth + widgetMargins + widgetPaddings;
                return cellWidth;
            };

            gridLayout.$connector.getAvailableWidth = (cell) => {
                const colspan = 1 + cell.x2 - cell.x1;
                let width = gridLayout.$connector.columnWidths[cell.x1];
                for (let i = 1; i < colspan; i++) {
                    width += gridLayout.$connector.columnWidths[cell.x1 + i] + gridLayout.$connector.getHorizontalSpacing();
                }
                return width;
            };

            gridLayout.$connector.getAvailableHeight = (cell) => {
                const rowspan = 1 + cell.y2 - cell.y1;
                let height = gridLayout.$connector.rowHeights[cell.y1];
                for (let i = 1; i < rowspan; i++) {
                    height += gridLayout.$connector.rowHeights[cell.y1 + i] + gridLayout.$connector.getVerticalSpacing();
                }
                return height;
            };

            // grid
            gridLayout.$connector.mutationObserver.observe(gridLayout, { childList: true });
            gridLayout.$connector.resizeObserver.observe(gridLayout);

            for (const cell of gridLayout.children) {
                const component = cell.firstChild;
                gridLayout.$connector.resizeObserver.observe(component);
            }

        }
    };
})();
