/*
 * Copyright 2000-2026 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.flow.component.ai.provider;

import java.io.Serializable;
import java.util.List;

import com.vaadin.flow.component.ai.common.AiAttachment;

import reactor.core.publisher.Flux;

/**
 * Framework-agnostic interface for Large Language Model (LLM) providers. This
 * interface enables AI-powered components to communicate with LLMs without
 * being tied to a specific implementation. Implementations are responsible for
 * managing conversation memory, handling streaming responses, processing
 * vendor-specific tool annotations, and handling file attachments.
 *
 * @author Vaadin Ltd.
 */
public interface LLMProvider {

    /**
     * Streams a response from the LLM based on the provided request. This
     * method returns a reactive stream that emits response tokens as they
     * become available from the LLM. The provider manages conversation history
     * internally, so each call to this method adds to the ongoing conversation
     * context.
     *
     * @param request
     *            the LLM request containing user message, system prompt,
     *            attachments, and tools, not {@code null}
     * @return a Flux stream that emits response tokens as strings, never
     *         {@code null}
     * @throws NullPointerException
     *             if request is {@code null}
     */
    Flux<String> stream(LLMRequest request);

    /**
     * Represents a request to the LLM containing all necessary context,
     * configuration, and tools. Requests are immutable.
     */
    interface LLMRequest extends Serializable {
        /**
         * Gets the user's message.
         *
         * @return the user message, never {@code null}
         */
        String userMessage();

        /**
         * Gets the list of file attachments to include with the request.
         * Attachments can be images, PDFs, text files, or other supported
         * formats that the LLM can analyze and reference in its response.
         *
         * @return the list of attachments, never {@code null} but may be empty
         */
        List<AiAttachment> attachments();

        /**
         * Gets the system prompt for this specific request. The system prompt
         * defines the LLM's behavior, role, and constraints. If {@code null},
         * the provider may use its own internal default system prompt.
         *
         * @return the system prompt, or {@code null} if not specified
         */
        String systemPrompt();

        /**
         * Gets the tool objects for this request. Tool objects are classes with
         * vendor-specific annotations (e.g., LangChain4j's {@code @Tool},
         * Spring AI's {@code @Tool}) that the provider can introspect and
         * convert to native tool definitions.
         *
         * @return array of tool objects, never {@code null} but may be empty
         */
        Object[] tools();
    }
}
