/*
 * Decompiled with CFR 0.152.
 */
package com.vaadin.flow.component.ai.orchestrator;

import com.vaadin.flow.component.ComponentEventListener;
import com.vaadin.flow.component.UI;
import com.vaadin.flow.component.ai.component.AiFileReceiver;
import com.vaadin.flow.component.ai.component.AiInput;
import com.vaadin.flow.component.ai.component.AiMessage;
import com.vaadin.flow.component.ai.component.AiMessageList;
import com.vaadin.flow.component.ai.component.InputSubmitListener;
import com.vaadin.flow.component.ai.provider.LLMProvider;
import com.vaadin.flow.component.messages.MessageInput;
import com.vaadin.flow.component.messages.MessageList;
import com.vaadin.flow.component.messages.MessageListItem;
import com.vaadin.flow.component.upload.UploadManager;
import com.vaadin.flow.server.Command;
import com.vaadin.flow.server.streams.InMemoryUploadCallback;
import com.vaadin.flow.server.streams.UploadHandler;
import java.io.Serializable;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Flux;

public class AiOrchestrator {
    private static final Logger LOGGER = LoggerFactory.getLogger(AiOrchestrator.class);
    private static final int TIMEOUT_SECONDS = 600;
    private final LLMProvider provider;
    private final String systemPrompt;
    private AiMessageList messageList;
    private AiInput input;
    private AiFileReceiver fileReceiver;
    private final List<LLMProvider.Attachment> pendingAttachments = new CopyOnWriteArrayList<LLMProvider.Attachment>();
    private Object[] tools = new Object[0];
    private String userName;
    private String aiName;
    private final AtomicBoolean isProcessing = new AtomicBoolean(false);

    private AiOrchestrator(LLMProvider provider, String systemPrompt) {
        Objects.requireNonNull(provider, "Provider cannot be null");
        this.provider = provider;
        this.systemPrompt = systemPrompt;
    }

    public static Builder builder(LLMProvider provider) {
        return new Builder(provider, null);
    }

    public static Builder builder(LLMProvider provider, String systemPrompt) {
        return new Builder(provider, systemPrompt);
    }

    public void prompt(String userMessage) {
        this.doPrompt(userMessage);
    }

    private void addUserMessageToList(String userMessage) {
        if (this.messageList != null) {
            AiMessage userItem = this.messageList.createMessage(userMessage, this.userName);
            this.messageList.addMessage(userItem);
        }
    }

    private AiMessage createAssistantMessagePlaceholder() {
        if (this.messageList == null) {
            return null;
        }
        AiMessage assistantMessage = this.messageList.createMessage("", this.aiName);
        this.messageList.addMessage(assistantMessage);
        return assistantMessage;
    }

    private void streamResponseToMessage(LLMProvider.LLMRequest request, AiMessage assistantMessage, UI ui) {
        Flux responseStream = this.provider.stream(request).timeout(Duration.ofSeconds(600L));
        responseStream.doFinally(signal -> this.isProcessing.set(false)).subscribe(token -> {
            if (assistantMessage != null && this.messageList != null) {
                ui.access((Command & Serializable)() -> assistantMessage.appendText((String)token));
            }
        }, error -> {
            String userMessage;
            if (error instanceof TimeoutException) {
                userMessage = "Request timed out. Please try again.";
                LOGGER.warn("LLM request timed out after {} seconds", (Object)600);
            } else {
                userMessage = "An error occurred. Please try again.";
                LOGGER.error("Error during LLM streaming", error);
            }
            if (assistantMessage != null && this.messageList != null) {
                ui.access((Command & Serializable)() -> assistantMessage.setText(userMessage));
            }
        }, () -> LOGGER.debug("LLM streaming completed successfully"));
    }

    private void doPrompt(String userMessage) {
        if (userMessage == null || userMessage.isBlank()) {
            return;
        }
        if (!this.isProcessing.compareAndSet(false, true)) {
            LOGGER.warn("Ignoring prompt: another request is already in progress");
            return;
        }
        this.processUserInput(userMessage);
    }

    private void processUserInput(final String userMessage) {
        UI ui = UI.getCurrentOrThrow();
        this.addUserMessageToList(userMessage);
        AiMessage assistantMessage = this.createAssistantMessagePlaceholder();
        String effectiveSystemPrompt = null;
        if (this.systemPrompt != null && !this.systemPrompt.isBlank()) {
            effectiveSystemPrompt = this.systemPrompt.trim();
        }
        final String finalSystemPrompt = effectiveSystemPrompt;
        final List attachments = this.pendingAttachments.stream().toList();
        this.clearPendingAttachments(ui);
        LLMProvider.LLMRequest request = new LLMProvider.LLMRequest(){

            @Override
            public String userMessage() {
                return userMessage;
            }

            @Override
            public List<LLMProvider.Attachment> attachments() {
                return attachments;
            }

            @Override
            public String systemPrompt() {
                return finalSystemPrompt;
            }

            @Override
            public Object[] tools() {
                return AiOrchestrator.this.tools;
            }
        };
        LOGGER.debug("Processing prompt with {} attachments", (Object)attachments.size());
        this.streamResponseToMessage(request, assistantMessage, ui);
    }

    private void clearPendingAttachments(UI ui) {
        this.pendingAttachments.clear();
        if (this.fileReceiver != null) {
            ui.access((Command & Serializable)() -> this.fileReceiver.clearFileList());
        }
    }

    private void configureFileReceiver() {
        if (this.fileReceiver == null) {
            return;
        }
        this.fileReceiver.setUploadHandler((UploadHandler)UploadHandler.inMemory((InMemoryUploadCallback & Serializable)(meta, data) -> {
            boolean isDuplicate = this.pendingAttachments.stream().anyMatch(a -> a.fileName().equals(meta.fileName()));
            if (isDuplicate) {
                throw new IllegalArgumentException("Duplicate file name: " + meta.fileName());
            }
            this.pendingAttachments.add(new LLMProvider.Attachment(){

                @Override
                public String fileName() {
                    return meta.fileName();
                }

                @Override
                public String contentType() {
                    return meta.contentType();
                }

                @Override
                public byte[] data() {
                    return data;
                }
            });
            LOGGER.debug("Added attachment: {}", (Object)meta.fileName());
        }));
        this.fileReceiver.addFileRemovedListener(fileName -> {
            boolean removed = this.pendingAttachments.removeIf(a -> a.fileName().equals(fileName));
            if (removed) {
                LOGGER.debug("Removed attachment: {}", fileName);
            }
        });
    }

    public static class Builder {
        private final LLMProvider provider;
        private final String systemPrompt;
        private AiMessageList messageList;
        private AiInput input;
        private AiFileReceiver fileReceiver;
        private Object[] tools = new Object[0];
        private String userName;
        private String aiName;

        private Builder(LLMProvider provider, String systemPrompt) {
            Objects.requireNonNull(provider, "Provider cannot be null");
            this.provider = provider;
            this.systemPrompt = systemPrompt;
        }

        public Builder withMessageList(AiMessageList messageList) {
            this.messageList = messageList;
            return this;
        }

        public Builder withMessageList(MessageList messageList) {
            this.messageList = Builder.wrapMessageList(messageList);
            return this;
        }

        public Builder withInput(AiInput input) {
            this.input = input;
            return this;
        }

        public Builder withInput(MessageInput messageInput) {
            this.input = Builder.wrapInput(messageInput);
            return this;
        }

        public Builder withFileReceiver(AiFileReceiver fileReceiver) {
            this.fileReceiver = fileReceiver;
            return this;
        }

        public Builder withFileReceiver(UploadManager uploadManager) {
            this.fileReceiver = Builder.wrapUploadManager(uploadManager);
            return this;
        }

        public Builder withTools(Object ... tools) {
            this.tools = tools != null ? tools : new Object[]{};
            return this;
        }

        public Builder withUserName(String userName) {
            Objects.requireNonNull(userName, "User name cannot be null");
            this.userName = userName;
            return this;
        }

        public Builder withAiName(String aiName) {
            Objects.requireNonNull(aiName, "AI name cannot be null");
            this.aiName = aiName;
            return this;
        }

        public AiOrchestrator build() {
            AiOrchestrator orchestrator = new AiOrchestrator(this.provider, this.systemPrompt);
            orchestrator.messageList = this.messageList;
            orchestrator.input = this.input;
            orchestrator.fileReceiver = this.fileReceiver;
            orchestrator.tools = this.tools == null ? new Object[]{} : this.tools;
            orchestrator.userName = this.userName == null ? "You" : this.userName;
            String string = orchestrator.aiName = this.aiName == null ? "Assistant" : this.aiName;
            if (this.input != null) {
                this.input.addSubmitListener(e -> orchestrator.doPrompt(e.getValue()));
            }
            if (this.fileReceiver != null) {
                orchestrator.configureFileReceiver();
            }
            LOGGER.debug("Built AiOrchestrator with messageList={}, input={}, fileReceiver={}, tools={}, userName={}, aiName={}", new Object[]{orchestrator.messageList != null, orchestrator.input != null, orchestrator.fileReceiver != null, orchestrator.tools.length, orchestrator.userName, orchestrator.aiName});
            return orchestrator;
        }

        private static AiMessageList wrapMessageList(MessageList messageList) {
            return new MessageListWrapper(messageList);
        }

        private static AiInput wrapInput(MessageInput messageInput) {
            return new MessageInputWrapper(messageInput);
        }

        private static AiFileReceiver wrapUploadManager(UploadManager uploadManager) {
            return new UploadManagerWrapper(uploadManager);
        }
    }

    private static class MessageListItemWrapper
    implements AiMessage {
        private final MessageListItem item;

        MessageListItemWrapper(String text, String userName) {
            this.item = new MessageListItem(text, Instant.now(), userName);
        }

        MessageListItem getItem() {
            return this.item;
        }

        @Override
        public String getText() {
            return this.item.getText();
        }

        @Override
        public void setText(String text) {
            this.item.setText(text);
        }

        @Override
        public Instant getTime() {
            return this.item.getTime();
        }

        @Override
        public String getUserName() {
            return this.item.getUserName();
        }

        @Override
        public void appendText(String token) {
            this.item.appendText(token);
        }
    }

    private record UploadManagerWrapper(UploadManager uploadManager) implements AiFileReceiver
    {
        @Override
        public void setUploadHandler(UploadHandler uploadHandler) {
            this.uploadManager.setUploadHandler(uploadHandler);
        }

        @Override
        public void addFileRemovedListener(Consumer<String> listener) {
            this.uploadManager.addFileRemovedListener((ComponentEventListener & Serializable)event -> listener.accept(event.getFileName()));
        }

        @Override
        public void clearFileList() {
            this.uploadManager.clearFileList();
        }
    }

    private record MessageInputWrapper(MessageInput messageInput) implements AiInput
    {
        @Override
        public void addSubmitListener(InputSubmitListener listener) {
            this.messageInput.addSubmitListener((ComponentEventListener & Serializable)event -> listener.onSubmit(() -> ((MessageInput.SubmitEvent)event).getValue()));
        }
    }

    private record MessageListWrapper(MessageList messageList) implements AiMessageList
    {
        @Override
        public void addMessage(AiMessage message) {
            this.messageList.addItem(((MessageListItemWrapper)message).getItem());
        }

        @Override
        public AiMessage createMessage(String text, String userName) {
            return new MessageListItemWrapper(text, userName);
        }
    }
}

