package com.vaadin.uitest.ai.services.vectordb;

import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.MediaType;
import org.springframework.web.reactive.function.client.WebClient;

import com.vaadin.uitest.ai.utils.KeysUtils;
import com.vaadin.uitest.model.vectordb.Document;
import com.vaadin.uitest.model.vectordb.Match;
import com.vaadin.uitest.model.vectordb.QueryResponse;

import reactor.core.publisher.Mono;

public class PineconeService {

    private static final Logger LOGGER = LoggerFactory
            .getLogger(PineconeService.class);

    private String PINECONE_API_KEY = KeysUtils.getPineconeKey();

    private String PINECONE_API_URL = KeysUtils.getPineconeUrl();

    private WebClient webClient;

    public PineconeService() {
        this.webClient = WebClient.builder().baseUrl(PINECONE_API_URL)
                .defaultHeader("Api-Key", PINECONE_API_KEY)
                .defaultHeader("Accept", MediaType.APPLICATION_JSON_VALUE)
                .defaultHeader("Content-Type", MediaType.APPLICATION_JSON_VALUE)
                .build();
    }

    public Mono<DocumentsQueryResult> findSimilarDocuments(
            List<Double> embedding, Namespace namespace, int topK) {
        if (Boolean.getBoolean("ai.debug")) {
            LOGGER.info("Finding similar documents in namespace: {}",
                    namespace.getLabel());
        }
        var body = Map.of("vector", embedding, "topK", topK, "includeMetadata",
                true, "namespace", namespace.getLabel());
        return getQueryResults(body, namespace);
    }

    private Mono<DocumentsQueryResult> getQueryResults(Map<String, Object> body,
            Namespace namespace) {
        return this.webClient.post().uri("/query").bodyValue(body).retrieve()
                .bodyToMono(QueryResponse.class).map(QueryResponse::getMatches)
                .flatMapIterable(matches -> matches)
                .filter(match -> match.getScore() > namespace
                        .getMatchScoreThreshold())
                .map(match -> getDocumentForTemplate(match,
                        namespace.getDocumentTemplate()))
                .filter(Optional::isPresent).map(Optional::get)
                .filter(doc -> !doc.isBlank()).collectList()
                .map(docs -> new DocumentsQueryResult(namespace, docs));
    }

    private Optional<Document> getDocumentForTemplate(Match match,
            DocumentTemplate documentTemplate) {
        if (match.getMetadata() == null || documentTemplate.getRequiredFields()
                .stream()
                .anyMatch(requiredField -> !match.getMetadata()
                        .containsKey(requiredField)
                        || match.getMetadata().get(requiredField).isBlank())) {
            return Optional.empty();
        }
        String content = documentTemplate.getContentTemplate()
                .formatted(documentTemplate.getRequiredFields().stream()
                        .map(match.getMetadata()::get).toArray());
        Document document = new Document();
        document.setContent(content);
        document.setScore(match.getScore().floatValue());
        document.setLabel(documentTemplate.getLabel());
        documentTemplate.getLinkKey().ifPresent(
                key -> document.setLink(match.getMetadata().get(key)));
        return Optional.of(document);
    }

}
