package com.vaadin.uitest.ai.services;

import java.time.Duration;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vaadin.uitest.ai.AiArguments;
import com.vaadin.uitest.ai.LLMService;
import com.vaadin.uitest.ai.utils.KeysUtils;
import com.vaadin.uitest.model.chat.ChatCompletionMessage;

import dev.langchain4j.model.openai.OpenAiChatModel;
import reactor.core.publisher.Flux;
import reactor.core.scheduler.Schedulers;

public abstract class DefaultLLMService implements LLMService {

    private static final Logger LOGGER = LoggerFactory
            .getLogger(DefaultLLMService.class);

    public DefaultLLMService() {
        LOGGER.info(
                "Using the {} model with {} of temperature {} of max-tokens, and a timeout of {} secs.",
                getModel(), getTemperature(), getMaxTokens(),
                LLMService.getTimeout());

    }

    @Override
    public String requestAI(AiArguments aiArguments) {
        return OpenAiChatModel.builder().apiKey(KeysUtils.getOpenAiKey())
                .modelName(aiArguments.getModel()).maxTokens(getMaxTokens())
                .temperature(getTemperature())
                .timeout(Duration.ofSeconds(LLMService.getTimeout())).build()
                .generate(aiArguments.getPrompt());
    }

    protected static String fluxToString(Flux<ChatCompletionMessage> flux) {
        try {
            // Specify the timeout duration
            long timeoutDuration = 60; // Timeout duration in seconds

            return flux.collectList()
                    .map(completion -> completion.stream()
                            .map(ChatCompletionMessage::getContent)
                            .collect(Collectors.joining("")))
                    .subscribeOn(Schedulers.boundedElastic()).toFuture()
                    .get(timeoutDuration, TimeUnit.SECONDS); // Apply the
                                                             // timeout here
        } catch (ExecutionException | InterruptedException
                | TimeoutException e) {
            LOGGER.error("Error while waiting for the completion", e);
            throw new RuntimeException("Operation timed out or was interrupted",
                    e);
        }
    }
}
