package com.vaadin.uitest.ai.services;

import static com.vaadin.uitest.ai.services.AiServiceConstants.MODEL;

import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vaadin.uitest.ai.AiArguments;
import com.vaadin.uitest.ai.LLMService;
import com.vaadin.uitest.ai.prompts.AiSnippets;
import com.vaadin.uitest.ai.services.vectordb.DocsAssistantService;
import com.vaadin.uitest.ai.services.vectordb.OpenAIService;
import com.vaadin.uitest.ai.services.vectordb.PineconeService;
import com.vaadin.uitest.model.TestFramework;
import com.vaadin.uitest.model.chat.ChatCompletionMessage;
import com.vaadin.uitest.model.chat.ChatCompletionMessage.Role;
import com.vaadin.uitest.model.codesnippetgeneration.CodeSnippet;

import reactor.core.publisher.Flux;

public class AiVectorDBGeneratorService implements LLMService {

    private static final Logger LOGGER = LoggerFactory
            .getLogger(AiVectorDBGeneratorService.class);

    private final DocsAssistantService docsAssistantService;

    public AiVectorDBGeneratorService() {
        LOGGER.info("Using the {} model", getModel());
        OpenAIService openAIService = new OpenAIService();
        PineconeService pineconeService = new PineconeService();
        this.docsAssistantService = new DocsAssistantService(openAIService,
                pineconeService);
    }

    @Override
    public String getPromptTemplate(AiArguments aiArguments) {
        TestFramework testFramework = aiArguments.getTestFramework();
        if (testFramework == null) {
            testFramework = aiArguments.getFramework()
                    .getDefaultTestFramework();
        }

        List<CodeSnippet> codeSnippets = AiSnippets
                .getCodeSnippets(aiArguments);
        String codeSnippetsArg = AiSnippets.getCodeSnippetsArg(codeSnippets);

        List<Object> arguments = new ArrayList<>();
        StringBuilder promptBuilder = new StringBuilder();
        if (TestFramework.PLAYWRIGHT_NODE.equals(testFramework)) {
            promptBuilder
                    .append(prompts().get(GENERATE_HEADING_PLAYWRIGHT_REACT));
            arguments.add(aiArguments.getFramework().getLabel());
            arguments.add(aiArguments.getGherkin());
            arguments.add(aiArguments.getViewUrl());
        } else {
            promptBuilder
                    .append(prompts().get(GENERATE_HEADING_PLAYWRIGHT_JAVA));
            arguments.add(aiArguments.getFramework().getLabel());
            arguments.add(aiArguments.getGherkin());
            arguments.add(aiArguments.getTestClassName());
            arguments.add(aiArguments.getViewUrl());
        }
        promptBuilder.append(prompts().get(GENERATE_IMPORTS));
        arguments.add(AiSnippets.getImportsArg(testFramework, codeSnippets));
        promptBuilder.append(prompts().get(GENERATE_SNIPPETS));
        arguments.add(codeSnippetsArg);

        return String.format(promptBuilder.toString(), arguments.toArray());
    }

    @Override
    public String requestAI(AiArguments aiArguments) {
        List<ChatCompletionMessage> history = new ArrayList<>();
        history.add(new ChatCompletionMessage(Role.ASSISTANT,
                getChatAssistantSystemMessage("Bart",
                        aiArguments.getFramework())));
        history.add(
                new ChatCompletionMessage(Role.USER, aiArguments.getPrompt()));

        String id = UUID.randomUUID().toString();
        Flux<ChatCompletionMessage> flux = docsAssistantService
                .getCompletionStream(history, aiArguments.getFramework(),
                        aiArguments.getGherkin(), id, false);
        return DefaultLLMService.fluxToString(flux);
    }

    @Override
    public String getModel() {
        return MODEL;
    }

}
