package com.vaadin.uitest.parser;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vaadin.uitest.ai.AiArguments;
import com.vaadin.uitest.ai.LLMService;
import com.vaadin.uitest.ai.utils.PromptUtils;
import com.vaadin.uitest.browser.Browser;
import com.vaadin.uitest.browser.BrowserLogin;
import com.vaadin.uitest.browser.ChromeBrowser;
import com.vaadin.uitest.browser.ChromeLogin;
import com.vaadin.uitest.model.Framework;
import com.vaadin.uitest.model.UiRoute;
import com.vaadin.uitest.model.scenario.TestScenario;
import com.vaadin.uitest.model.scenario.TestScenarioStep;
import com.vaadin.uitest.model.scenario.TestScenarioStepType;
import com.vaadin.uitest.model.scenario.TestScenarios;

public abstract class ParserLLM<T extends LLMService> implements Parser {

    private static final Logger LOGGER = LoggerFactory
            .getLogger(ParserLLM.class);

    protected final String baseUrl;

    public ParserLLM() {
        this(null);
    }

    public ParserLLM(String baseUrl) {
        this.baseUrl = baseUrl;
    }

    abstract T getService();

    @Override
    public void generateTestScenarios(UiRoute view) {

        AiArguments aiArguments = new AiArguments();
        aiArguments.setSource(view.getSource());
        aiArguments.setHtml(view.getHtml());
        aiArguments.setFramework(Framework.getByValue(view.getFramework()));
        aiArguments.setModel(getService().getModel());

        String prompt = getService().getPromptTemplate(aiArguments);
        aiArguments.setPrompt(prompt);

        LOGGER.info(
                "Parsing the view {} ({}) Source: {} Bytes - {} Words, Html: {} Bytes - {} Words, Prompt: {} Bytes - {} Words",
                view.getSimpleName(), view.getRoute(),
                view.getSource().length(),
                view.getSource().split("[^\\w]+").length,
                view.getHtml().length(), view.getHtml().split("[^\\w]").length,
                prompt.length(), prompt.split("[^\\\\w]").length);

        String resp = getService().getGeneratedResponse(aiArguments);
        view.setGherkin(resp);
        TestScenarios testScenarios = TestScenarios.parse(resp);
        view.setTestScenarios(testScenarios);
    }

    @Override
    public void getViewHtml(String cssSelector, String loginHtml,
            String loginRoute, UiRoute view) {
        if (baseUrl != null) {
            String url = baseUrl + "/" + view.getRoute();
            LOGGER.info("Getting Html for view={} from url={}",
                    view.getClassName(), url);
            Browser chromeBrowser = getBrowser();
            try {
                String html = url.equals(loginRoute)
                        // for the login UI reuse html found above
                        ? loginHtml
                        // otherwise visit the url
                        : chromeBrowser.getHTMLContent(url, cssSelector);
                html = PromptUtils.cleanHtml(html);
                view.setHtml(html);
            } finally {
                chromeBrowser.close();
            }
        }
    }

    @Override
    public UiRoute updateTestScenarios(UiRoute view) {
        if (baseUrl == null || !Boolean.getBoolean("recursive")) {
            return view;
        }

        String url = view.getBaseUrl() + view.getRoute();
        String initialDomState = view.getHtml();
        view.getTestScenarios().getScenarios().forEach(testScenario -> {
            updateGherkinScenario(url, initialDomState, testScenario,
                    Framework.getByValue(view.getFramework()));
        });
        return view;
    }

    protected void updateGherkinScenario(String url, String initialDomState,
            TestScenario gherkinTestScenario, Framework framework) {
        TestScenarioStep action = gherkinTestScenario.getSteps().stream()
                .filter(s -> s.getType() == TestScenarioStepType.ACTION)
                .findAny().orElse(null);
        TestScenarioStep assertion = gherkinTestScenario.getSteps().stream()
                .filter(s -> s.getType() == TestScenarioStepType.ASSERTION)
                .findAny().orElse(null);
        if (assertion == null || action == null) {
            return;
        }

        Browser browser = getBrowser();
        List<String> scripts = new ArrayList<String>();
        try {
            LOGGER.info("Executing action in the view: {}",
                    gherkinTestScenario);
            scripts = gherkinTestScenario.getSteps().stream()
                    .filter(s -> s.getJsSnippet() != null
                            && s.getType() == TestScenarioStepType.ACTION)
                    .map(TestScenarioStep::getJsSnippet)
                    .collect(Collectors.toList());
            String result = browser.getHTMLContent(url, scripts);
            action.setResultHtml(PromptUtils.cleanHtml(result));
        } catch (Exception e) {
            LOGGER.error(
                    "!! Ignoring action, Exception when visiting URL={} JS={} {}",
                    url, scripts, e.getMessage().split("\\R")[0]);
        } finally {
            browser.close();
        }

    }

    @Override
    public String getBaseUrl() {
        return baseUrl;
    }

    @Override
    public BrowserLogin getBrowserLogin() {
        return new ChromeLogin();
    }

    @Override
    public Browser getBrowser() {
        return new ChromeBrowser();
    }

}
