package com.vaadin.uitest.model.flow;

import com.vaadin.uitest.model.codesnippetgeneration.Action;
import com.vaadin.uitest.model.codesnippetgeneration.Assertion;
import com.vaadin.uitest.model.codesnippetgeneration.ElementProperty;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.stream.Collectors;

public enum FlowComponentElement {

    BUTTON,

    CHECKBOX,

    CHECKBOX_GROUP,

    CHECKBOX_GROUP_ITEM,

    COMBO_BOX,

    COMBO_BOX_ITEM,

    CONFIRM_DIALOG,

    DATE_PICKER,

    DATE_TIME_PICKER,

    DIALOG,

    EMAIL_FIELD,

    GRID,

    GRID_COLUMN,

    GRID_ITEM,

    GRID_CELL,

    INTEGER_FIELD,

    LIST_BOX,

    LIST_BOX_ITEM,

    MENU_BAR,

    MENU_BAR_ITEM,

    MULTI_SELECT_COMBO_BOX,

    MULTI_SELECT_COMBO_BOX_ITEM,

    MULTI_SELECT_LIST_BOX,

    MULTI_SELECT_LIST_BOX_ITEM,

    NOTIFICATION,

    NUMBER_FIELD,

    PASSWORD_FIELD,

    RADIO_BUTTON_GROUP,

    RADIO_BUTTON,

    SELECT,

    SELECT_ITEM,

    SIDE_NAV,

    SIDE_NAV_ITEM,

    TABS,

    TAB,

    TEXT_AREA,

    TEXT_FIELD,

    TIME_PICKER,

    VIRTUAL_LIST,

    VIRTUAL_LIST_ITEM,

    ELEMENT;

    public Set<ElementProperty> getElementProperties() {
        Set<ElementProperty> elementProperties = new HashSet<>(
                getCommonElementProperties());
        elementProperties.addAll(getSpecialElementProperties(this));
        return elementProperties;
    }

    public Set<Action> getAllowedActions() {
        return getAllowedActions(this);
    }

    public Set<Assertion> getAllowedAssertions() {
        Set<Assertion> allowedAssertions = new HashSet<>(
                getCommonAllowedAssertions());
        allowedAssertions.addAll(getSpecialAllowedAssertions(this));
        return allowedAssertions;
    }

    public static FlowComponentElement getByName(String name) {
        return Arrays.stream(values())
                .filter(element -> element.name().equals(name)).findAny()
                .orElse(null);
    }

    public static FlowComponentElement getCorrespondingElement(
            FlowComponent component) {
        return getByName(component.name());
    }

    private static Set<ElementProperty> getSpecialElementProperties(
            FlowComponentElement property) {
        switch (property) {
        case BUTTON, CHECKBOX, CHECKBOX_GROUP, CHECKBOX_GROUP_ITEM,
                COMBO_BOX_ITEM, LIST_BOX_ITEM, MENU_BAR_ITEM,
                MULTI_SELECT_COMBO_BOX_ITEM, MULTI_SELECT_LIST_BOX_ITEM,
                RADIO_BUTTON_GROUP, RADIO_BUTTON, SELECT_ITEM, SIDE_NAV_ITEM,
                TAB, VIRTUAL_LIST, VIRTUAL_LIST_ITEM -> {
            return Set.of(ElementProperty.LABEL);
        }
        case COMBO_BOX, DATE_PICKER, DATE_TIME_PICKER, EMAIL_FIELD,
                INTEGER_FIELD, LIST_BOX, MULTI_SELECT_COMBO_BOX,
                MULTI_SELECT_LIST_BOX, NUMBER_FIELD, PASSWORD_FIELD, SELECT,
                TEXT_AREA, TEXT_FIELD, TIME_PICKER -> {
            return Set.of(ElementProperty.LABEL, ElementProperty.PLACEHOLDER);
        }
        case CONFIRM_DIALOG, DIALOG, NOTIFICATION, SIDE_NAV, TABS -> {
            return Set.of(ElementProperty.TITLE);
        }
        case GRID, GRID_COLUMN, GRID_ITEM, GRID_CELL, MENU_BAR -> {
            return Set.of();
        }
        case ELEMENT -> {
            return Arrays.stream(ElementProperty.values())
                    .collect(Collectors.toSet());
        }
        }
        return Collections.emptySet();
    }

    private static Set<ElementProperty> getCommonElementProperties() {
        return Set.of(ElementProperty.ROLE, ElementProperty.TEXT,
                ElementProperty.NOT_TEXT, ElementProperty.TAG_NAME);
    }

    private static Set<Action> getAllowedActions(
            FlowComponentElement property) {
        switch (property) {
        case BUTTON, COMBO_BOX_ITEM, LIST_BOX_ITEM, MENU_BAR_ITEM,
                MULTI_SELECT_COMBO_BOX_ITEM, MULTI_SELECT_LIST_BOX_ITEM,
                RADIO_BUTTON, SELECT_ITEM, SIDE_NAV_ITEM, VIRTUAL_LIST_ITEM -> {
            return Set.of(Action.CLICK);
        }
        case CHECKBOX, CHECKBOX_GROUP_ITEM -> {
            return Set.of(Action.CHECK, Action.UNCHECK);
        }
        case CHECKBOX_GROUP, LIST_BOX, MENU_BAR, MULTI_SELECT_COMBO_BOX,
                MULTI_SELECT_LIST_BOX, RADIO_BUTTON_GROUP, SELECT, SIDE_NAV,
                TABS, VIRTUAL_LIST -> {
            return Set.of(Action.SELECT_OPTION);
        }
        case COMBO_BOX, DATE_PICKER, DATE_TIME_PICKER, TIME_PICKER -> {
            return Set.of(Action.SELECT_OPTION, Action.FILL);
        }
        case CONFIRM_DIALOG, DIALOG, GRID, GRID_COLUMN, GRID_ITEM, NOTIFICATION,
                TAB -> {
            return Set.of();
        }
        case EMAIL_FIELD, GRID_CELL, INTEGER_FIELD, NUMBER_FIELD,
                PASSWORD_FIELD, TEXT_AREA, TEXT_FIELD -> {
            return Set.of(Action.FILL);
        }
        case ELEMENT -> {
            return Arrays.stream(Action.values()).collect(Collectors.toSet());
        }
        }
        return Collections.emptySet();
    }

    private static Set<Assertion> getSpecialAllowedAssertions(
            FlowComponentElement property) {
        switch (property) {
        case BUTTON, CHECKBOX_GROUP_ITEM, COMBO_BOX_ITEM, CONFIRM_DIALOG,
                DIALOG, GRID, GRID_COLUMN, GRID_ITEM, LIST_BOX_ITEM, MENU_BAR,
                MENU_BAR_ITEM, MULTI_SELECT_COMBO_BOX_ITEM,
                MULTI_SELECT_LIST_BOX_ITEM, NOTIFICATION, SELECT_ITEM,
                SIDE_NAV_ITEM, TAB, VIRTUAL_LIST_ITEM -> {
            return Set.of();
        }
        case CHECKBOX -> {
            return Set.of(Assertion.HAS_VALUE, Assertion.IS_CHECKED);
        }
        case CHECKBOX_GROUP, MULTI_SELECT_LIST_BOX -> {
            return Set.of(Assertion.HAS_VALUES);
        }
        case COMBO_BOX, DATE_PICKER, DATE_TIME_PICKER, EMAIL_FIELD,
                INTEGER_FIELD, NUMBER_FIELD, PASSWORD_FIELD, SELECT, TEXT_AREA,
                TEXT_FIELD, TIME_PICKER -> {
            return Set.of(Assertion.HAS_VALUE, Assertion.IS_EDITABLE);
        }
        case GRID_CELL -> {
            return Set.of(Assertion.IS_EDITABLE);
        }
        case LIST_BOX, RADIO_BUTTON_GROUP, RADIO_BUTTON, SIDE_NAV, TABS,
                VIRTUAL_LIST -> {
            return Set.of(Assertion.HAS_VALUE);
        }
        case MULTI_SELECT_COMBO_BOX -> {
            return Set.of(Assertion.HAS_VALUES, Assertion.IS_EDITABLE);
        }
        case ELEMENT -> {
            return Arrays.stream(Assertion.values())
                    .collect(Collectors.toSet());
        }
        }
        return Collections.emptySet();
    }

    private static Set<Assertion> getCommonAllowedAssertions() {
        return Set.of(Assertion.IS_VISIBLE, Assertion.IS_IN_VIEWPORT,
                Assertion.IS_HIDDEN, Assertion.IS_FOCUSED, Assertion.HAS_TEXT,
                Assertion.HAS_COUNT, Assertion.CONTAINS_TEXT,
                Assertion.IS_EMPTY, Assertion.IS_DISABLED,
                Assertion.IS_ENABLED);
    }
}
