package com.vaadin.uitest.model.codesnippetgeneration;

import com.vaadin.uitest.model.TestFramework;
import com.vaadin.uitest.model.flow.FlowComponentElement;
import com.vaadin.uitest.model.scenario.TestScenarioStep;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

public class CodeSnippetGenerationArguments {

    private List<GherkinElement> elementsInHierarchicalOrder;
    private InteractionArguments interactionArguments;
    private String originalGherkinStep;
    private TestFramework testFramework;

    public List<GherkinElement> getElementsInHierarchicalOrder() {
        return elementsInHierarchicalOrder;
    }

    public void setElementsInHierarchicalOrder(
            List<GherkinElement> elementsInHierarchicalOrder) {
        this.elementsInHierarchicalOrder = elementsInHierarchicalOrder;
    }

    public InteractionArguments getInteractionArguments() {
        return interactionArguments;
    }

    public void setInteractionArguments(
            InteractionArguments interactionArguments) {
        this.interactionArguments = interactionArguments;
    }

    public String getOriginalGherkinStep() {
        return originalGherkinStep;
    }

    public void setOriginalGherkinStep(String originalGherkinStep) {
        this.originalGherkinStep = originalGherkinStep;
    }

    public TestFramework getTestFramework() {
        return testFramework;
    }

    public void setTestFramework(TestFramework testFramework) {
        this.testFramework = testFramework;
    }

    // TODO optimize, polish and make more stable
    public static CodeSnippetGenerationArguments fromGherkinStep(
            TestScenarioStep scenarioStep) {
        if (scenarioStep == null) {
            return null;
        }
        CodeSnippetGenerationArguments codeSnippetGenerationArguments = new CodeSnippetGenerationArguments();
        codeSnippetGenerationArguments
                .setOriginalGherkinStep(scenarioStep.getDescription());
        codeSnippetGenerationArguments
                .setInteractionArguments(getInteractionArguments(scenarioStep));
        codeSnippetGenerationArguments.setElementsInHierarchicalOrder(
                List.of(getGherkinElement(scenarioStep.getDescription())));
        return codeSnippetGenerationArguments;
    }

    private static GherkinElement getGherkinElement(String description) {
        GherkinElement gherkinElement = new GherkinElement();
        gherkinElement
                .setFlowComponentElement(getFlowComponentElement(description));
        gherkinElement.setIdentifiers(getElementIdentifiers(description));
        return gherkinElement;
    }

    private static Map<ElementProperty, String> getElementIdentifiers(
            String description) {
        Map<ElementProperty, Set<String>> properties = new LinkedHashMap<>();
        properties.put(ElementProperty.ROLE, Set.of("role"));
        properties.put(ElementProperty.LABEL, Set.of("label"));
        properties.put(ElementProperty.TITLE, Set.of("title", "header"));
        properties.put(ElementProperty.ALT_TEXT, Set.of("alt text", "alttext"));
        properties.put(ElementProperty.TEXT, Set.of("text"));
        properties.put(ElementProperty.PLACEHOLDER, Set.of("placeholder"));
        properties.put(ElementProperty.TAG_NAME, Set.of("tag name", "tagname"));
        properties.put(ElementProperty.CSS, Set.of("css"));
        String descriptionLowerCase = description.toLowerCase();
        Map<ElementProperty, String> identifiers = new HashMap<>();
        properties.forEach((key, probablePropertyNames) -> probablePropertyNames
                .stream().map(name -> Set.of("with " + name, "and " + name))
                .flatMap(Set::stream).map(descriptionLowerCase::indexOf)
                .filter(index -> index != -1).findFirst().ifPresent(index -> {
                    String arg = getArg(description, index);
                    if (arg != null) {
                        identifiers.put(key, arg);
                    }
                }));
        return identifiers;
    }

    private static FlowComponentElement getFlowComponentElement(
            String description) {
        List<FlowComponentElement> elements = List.of(
                FlowComponentElement.VIRTUAL_LIST_ITEM,
                FlowComponentElement.VIRTUAL_LIST,
                FlowComponentElement.RADIO_BUTTON_GROUP,
                FlowComponentElement.RADIO_BUTTON,
                FlowComponentElement.SIDE_NAV_ITEM,
                FlowComponentElement.SIDE_NAV,
                FlowComponentElement.MULTI_SELECT_LIST_BOX_ITEM,
                FlowComponentElement.MULTI_SELECT_LIST_BOX,
                FlowComponentElement.LIST_BOX_ITEM,
                FlowComponentElement.LIST_BOX,
                FlowComponentElement.MULTI_SELECT_COMBO_BOX_ITEM,
                FlowComponentElement.MULTI_SELECT_COMBO_BOX,
                FlowComponentElement.COMBO_BOX_ITEM,
                FlowComponentElement.COMBO_BOX,
                FlowComponentElement.CHECKBOX_GROUP_ITEM,
                FlowComponentElement.CHECKBOX_GROUP,
                FlowComponentElement.CHECKBOX, FlowComponentElement.GRID_CELL,
                FlowComponentElement.GRID_ITEM,
                FlowComponentElement.GRID_COLUMN, FlowComponentElement.GRID,
                FlowComponentElement.CONFIRM_DIALOG,
                FlowComponentElement.DIALOG, FlowComponentElement.MENU_BAR_ITEM,
                FlowComponentElement.MENU_BAR,
                FlowComponentElement.DATE_TIME_PICKER,
                FlowComponentElement.TIME_PICKER,
                FlowComponentElement.DATE_PICKER, FlowComponentElement.BUTTON,
                FlowComponentElement.EMAIL_FIELD,
                FlowComponentElement.INTEGER_FIELD,
                FlowComponentElement.NOTIFICATION,
                FlowComponentElement.NUMBER_FIELD,
                FlowComponentElement.PASSWORD_FIELD,
                FlowComponentElement.TEXT_AREA, FlowComponentElement.TEXT_FIELD,
                FlowComponentElement.SELECT_ITEM, FlowComponentElement.SELECT,
                FlowComponentElement.TABS, FlowComponentElement.TAB);
        String descriptionLowerCase = description.toLowerCase();
        return elements.stream()
                .filter(e -> descriptionLowerCase
                        .contains(e.name().replace("_", " ").toLowerCase()))
                .findFirst().orElse(FlowComponentElement.ELEMENT);
    }

    private static InteractionArguments getInteractionArguments(
            TestScenarioStep scenarioStep) {
        switch (scenarioStep.getType()) {
        case ACTION -> {
            return getActionInteractionArguments(scenarioStep.getDescription());
        }
        case ASSERTION -> {
            return getAssertionInteractionArguments(
                    scenarioStep.getDescription());
        }
        }
        return null;
    }

    private static InteractionArguments getActionInteractionArguments(
            String description) {
        List<Action> actions = List.of(Action.UNCHECK, Action.CHECK,
                Action.DOUBLE_CLICK, Action.CLICK, Action.FILL,
                Action.PRESS_SEQUENTIALLY, Action.SELECT_OPTION,
                Action.SELECT_TEXT);
        String descriptionLowerCase = description.toLowerCase();
        for (Action action : actions) {
            Set<String> tags = getActionTags(action);
            Optional<Integer> index = getIndex(tags, descriptionLowerCase);
            if (index.isEmpty()) {
                continue;
            }
            InteractionArguments interactionArguments = new InteractionArguments();
            interactionArguments.setInteraction(action);
            if (hasArgument(action)) {
                String arg = getArg(description, index.get());
                if (arg != null) {
                    interactionArguments.setArgs(arg);
                }
            }
            return interactionArguments;
        }
        return null;
    }

    private static InteractionArguments getAssertionInteractionArguments(
            String description) {
        List<Assertion> assertions = List.of(Assertion.HAS_VALUES,
                Assertion.HAS_VALUE, Assertion.CONTAINS_TEXT,
                Assertion.HAS_TEXT, Assertion.IS_CHECKED, Assertion.IS_DISABLED,
                Assertion.IS_EDITABLE, Assertion.IS_ENABLED,
                Assertion.IS_FOCUSED, Assertion.IS_HIDDEN,
                Assertion.IS_IN_VIEWPORT, Assertion.IS_EMPTY,
                Assertion.IS_VISIBLE);
        String descriptionLowerCase = description.toLowerCase();
        for (Assertion assertion : assertions) {
            boolean isNegative = true;
            Set<String> tags = getAssertionTags(assertion, isNegative);
            Optional<Integer> index = getIndex(tags, descriptionLowerCase);
            if (index.isEmpty()) {
                isNegative = false;
                tags = getAssertionTags(assertion, isNegative);
                index = getIndex(tags, descriptionLowerCase);
            }
            if (index.isEmpty()) {
                continue;
            }
            InteractionArguments interactionArguments = new InteractionArguments();
            interactionArguments.setInteraction(assertion);
            interactionArguments.setNegative(isNegative);
            if (hasArgument(assertion)) {
                String arg = getArg(description, index.get());
                if (arg != null) {
                    interactionArguments.setArgs(arg);
                }
            }
            return interactionArguments;
        }
        return null;
    }

    private static Optional<Integer> getIndex(Set<String> tags,
            String descriptionLowerCase) {
        Optional<Integer> index;
        if (tags == null) {
            index = Optional.empty();
        } else {
            index = tags.stream().filter(descriptionLowerCase::contains)
                    .map(descriptionLowerCase::indexOf).findFirst();
        }
        return index;
    }

    private static String getArg(String description, int index) {
        int startIndexOfValue = description.indexOf("'", index);
        if (startIndexOfValue != -1) {
            int endIndexOfValue = description.indexOf("'",
                    startIndexOfValue + 1);
            if (endIndexOfValue != -1) {
                return description.substring(startIndexOfValue + 1,
                        endIndexOfValue);
            }
        }
        return null;
    }

    private static boolean hasArgument(Action action) {
        return Set
                .of(Action.FILL, Action.SELECT_OPTION,
                        Action.PRESS_SEQUENTIALLY, Action.SELECT_TEXT)
                .contains(action);
    }

    private static boolean hasArgument(Assertion assertion) {
        return Set.of(Assertion.HAS_VALUE, Assertion.HAS_VALUES,
                Assertion.HAS_TEXT, Assertion.HAS_COUNT,
                Assertion.CONTAINS_TEXT).contains(assertion);
    }

    private static Set<String> getActionTags(Action action) {
        switch (action) {
        case FILL -> {
            return Set.of("fills", "filling", "filled", "enters", "entering",
                    "entered");
        }
        case CHECK -> {
            return Set.of("checks", "checking", "checked");
        }
        case UNCHECK -> {
            return Set.of("unchecks", "unchecking", "unchecked");
        }
        case SELECT_OPTION -> {
            return Set.of("selects", "selecting", "selected");
        }
        case CLICK -> {
            return Set.of("clicks", "clicking", "clicked", "taps", "tapping",
                    "tapped");
        }
        case DOUBLE_CLICK -> {
            return Set.of("double clicks", "double clicking", "double clicked");
        }
        case PRESS_SEQUENTIALLY -> {
            return Set.of("presses", "pressing", "pressed", "types", "typing",
                    "typed");
        }
        case SELECT_TEXT -> {
            return Set.of("selects", "selecting", "selected", "highlights",
                    "highlighting", "highlighted");
        }
        }
        return null;
    }

    private static Set<String> getAssertionTags(Assertion assertion,
            boolean isNegative) {
        switch (assertion) {
        case CONTAINS_TEXT -> {
            return isNegative ? Set.of("not contain text")
                    : Set.of("contains text");
        }
        case HAS_TEXT -> {
            return isNegative ? Set.of("not have text") : Set.of("has text");
        }
        case HAS_VALUE -> {
            return isNegative ? Set.of("not have value") : Set.of("has value");
        }
        case HAS_VALUES -> {
            return isNegative ? Set.of("not have values")
                    : Set.of("has values");
        }
        case IS_CHECKED -> {
            return isNegative ? Set.of("not checked", "unchecked")
                    : Set.of("checked");
        }
        case IS_DISABLED -> {
            return isNegative ? Set.of("not disabled") : Set.of("disabled");
        }
        case IS_EDITABLE -> {
            return isNegative ? Set.of("not editable") : Set.of("editable");
        }
        case IS_EMPTY -> {
            return isNegative ? Set.of("not empty") : Set.of("empty");
        }
        case IS_ENABLED -> {
            return isNegative ? Set.of("not enabled") : Set.of("enabled");
        }
        case IS_HIDDEN -> {
            return isNegative ? Set.of("not hidden") : Set.of("hidden");
        }
        case IS_VISIBLE -> {
            return isNegative ? Set.of("not visible", "not see")
                    : Set.of("visible", " see");
        }
        }
        return null;
    }
}
