package com.vaadin.uitest.codesnippetgeneration;

import java.util.Arrays;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.vaadin.uitest.model.codesnippetgeneration.Action;
import com.vaadin.uitest.model.codesnippetgeneration.Assertion;
import com.vaadin.uitest.model.codesnippetgeneration.CodeSnippet;
import com.vaadin.uitest.model.codesnippetgeneration.CodeSnippetGenerator;
import com.vaadin.uitest.model.codesnippetgeneration.ElementProperty;
import com.vaadin.uitest.model.codesnippetgeneration.GherkinElement;
import com.vaadin.uitest.model.codesnippetgeneration.LocatedGherkinElement;
import com.vaadin.uitest.model.flow.FlowComponentElement;

public class PlaywrightNodeCodeSnippetGenerator
        implements CodeSnippetGenerator {

    @Override
    public CodeSnippet getLocatorCodeSnippet(GherkinElement gherkinElement,
            String elementVarName, String parentVarName) {
        String locatorCode = "";
        Map<ElementProperty, String> identifiers = gherkinElement
                .getIdentifiers();
        if (identifiers.containsKey(ElementProperty.ROLE)) {
            if (identifiers.containsKey(ElementProperty.LABEL)) {
                locatorCode = String.format(
                        "let %s = %s.getByRole('%s', { name: '%s' });",
                        elementVarName, parentVarName,
                        identifiers.get(ElementProperty.ROLE),
                        identifiers.get(ElementProperty.LABEL));
            } else {
                locatorCode = String.format("let %s = %s.getByRole('%s');",
                        elementVarName, parentVarName,
                        identifiers.get(ElementProperty.ROLE));
            }
        } else if (identifiers.containsKey(ElementProperty.LABEL)) {
            if (FlowComponentElement.BUTTON
                    .equals(gherkinElement.getFlowComponentElement())) {
                locatorCode = String.format(
                        "let %s = %s.getByRole('button', { name: '%s' });",
                        elementVarName, parentVarName,
                        identifiers.get(ElementProperty.LABEL));
            } else {
                locatorCode = String.format("let %s = %s.getByLabel('%s');",
                        elementVarName, parentVarName,
                        identifiers.get(ElementProperty.LABEL));
            }
        } else if (identifiers.containsKey(ElementProperty.PLACEHOLDER)) {
            locatorCode = String.format("let %s = %s.getByPlaceholder('%s');",
                    elementVarName, parentVarName,
                    identifiers.get(ElementProperty.PLACEHOLDER));
        } else if (identifiers.containsKey(ElementProperty.ALT_TEXT)) {
            locatorCode = String.format("let %s = %s.getByAltText('%s');",
                    elementVarName, parentVarName,
                    identifiers.get(ElementProperty.ALT_TEXT));
        } else if (identifiers.containsKey(ElementProperty.TITLE)) {
            locatorCode = String.format("let %s = %s.getByTitle('%s');",
                    elementVarName, parentVarName,
                    identifiers.get(ElementProperty.TITLE));
        } else if (identifiers.containsKey(ElementProperty.TAG_NAME)) {
            locatorCode = String.format("let %s = %s.locator('%s');",
                    elementVarName, parentVarName,
                    identifiers.get(ElementProperty.TAG_NAME));
        } else if (identifiers.containsKey(ElementProperty.CSS)) {
            locatorCode = String.format("let %s = %s.locator('%s');",
                    elementVarName, parentVarName,
                    identifiers.get(ElementProperty.CSS));
        } else if (identifiers.containsKey(ElementProperty.XPATH)) {
            locatorCode = String.format("let %s = %s.locator('//%s');",
                    elementVarName, parentVarName,
                    identifiers.get(ElementProperty.XPATH));
        }
        if (identifiers.containsKey(ElementProperty.TEXT)) {
            if (locatorCode.isEmpty()) {
                locatorCode = String.format("let %s = %s.getByText('%s');",
                        elementVarName, parentVarName,
                        identifiers.get(ElementProperty.TEXT));
            } else {
                locatorCode += String.format(
                        "\n%s = %s.filter({ hasText: '%s' });", elementVarName,
                        elementVarName, identifiers.get(ElementProperty.TEXT));
            }
        }
        if (!locatorCode.isEmpty()
                && identifiers.containsKey(ElementProperty.NOT_TEXT)) {
            locatorCode += String.format(
                    "\n%s = %s.filter({ hasNotText: '%s' });", elementVarName,
                    elementVarName, identifiers.get(ElementProperty.NOT_TEXT));
        }
        CodeSnippet locatorCodeSnippet = new CodeSnippet();
        locatorCodeSnippet.setCode(locatorCode);
        return locatorCodeSnippet;
    }

    @Override
    public CodeSnippet getActionCodeSnippet(LocatedGherkinElement element,
            Action action, String... args) {
        String actionCode = null;
        switch (action) {
        case FILL -> actionCode = String.format("await %s.fill('%s');",
                element.getVariableName(), args[0]);
        case CHECK -> actionCode = String.format("await %s.check();",
                element.getVariableName());
        case UNCHECK -> actionCode = String.format("await %s.uncheck();",
                element.getVariableName());
        case SELECT_OPTION -> {
            String optionsStr;
            if (args.length == 1) {
                optionsStr = "'" + args[0] + "'";
            } else {
                optionsStr = Arrays.stream(args)
                        .map(arg -> String.format("'%s'", arg))
                        .collect(Collectors.joining(", "));
            }
            actionCode = String.format("await %s.selectOption(%s);",
                    element.getVariableName(), optionsStr);
        }
        case CLICK -> actionCode = String.format("await %s.click();",
                element.getVariableName());
        case DOUBLE_CLICK -> actionCode = String.format("await %s.dblclick();",
                element.getVariableName());
        case PRESS_SEQUENTIALLY ->
            actionCode = String.format("await %s.pressSequentially('%s');",
                    element.getVariableName(), args[0]);
        case PRESS -> actionCode = String.format("await %s.press('%s');",
                element.getVariableName(), args[0]);
        case SELECT_TEXT -> {
            return null;
        }
        }
        CodeSnippet codeSnippet = new CodeSnippet();
        codeSnippet.setCode(actionCode);
        return codeSnippet;
    }

    @Override
    public CodeSnippet getAssertionCodeSnippet(LocatedGherkinElement element,
            Assertion assertion, boolean negative, String... args) {
        String code = null;
        String elementVariableName = element.getVariableName();
        String negativeSection = negative ? ".not" : "";
        switch (assertion) {
        case CONTAINS_TEXT -> {
            code = String.format("await expect(%s)%s.toContainText(%s);",
                    elementVariableName, negativeSection, getArray(args));
        }
        case HAS_COUNT -> {
            code = String.format("await expect(%s)%s.toHaveCount(%s);",
                    elementVariableName, negativeSection, args[0]);
        }
        case HAS_TEXT -> {
            code = String.format("await expect(%s)%s.toHaveText(%s);",
                    elementVariableName, negativeSection, getArray(args));
        }
        case HAS_VALUE -> {
            code = String.format("await expect(%s)%s.toHaveValue('%s');",
                    elementVariableName, negativeSection, args[0]);
        }
        case HAS_VALUES -> {
            code = String.format("await expect(%s)%s.toHaveValues(%s);",
                    elementVariableName, negativeSection, getArray(args));
        }
        case IS_CHECKED -> {
            code = String.format("await expect(%s)%s.toBeChecked();",
                    elementVariableName, negativeSection);
        }
        case IS_DISABLED -> {
            code = String.format("await expect(%s)%s.toBeDisabled();",
                    elementVariableName, negativeSection);
        }
        case IS_EDITABLE -> {
            code = String.format("await expect(%s)%s.toBeEditable();",
                    elementVariableName, negativeSection);
        }
        case IS_EMPTY -> {
            code = String.format("await expect(%s)%s.toBeEmpty();",
                    elementVariableName, negativeSection);
        }
        case IS_ENABLED -> {
            code = String.format("await expect(%s)%s.toBeEnabled();",
                    elementVariableName, negativeSection);
        }
        case IS_FOCUSED -> {
            code = String.format("await expect(%s)%s.toBeFocused();",
                    elementVariableName, negativeSection);
        }
        case IS_HIDDEN -> {
            code = String.format("await expect(%s)%s.toBeHidden();",
                    elementVariableName, negativeSection);
        }
        case IS_IN_VIEWPORT -> {
            code = String.format("await expect(%s)%s.toBeInViewport();",
                    elementVariableName, negativeSection);
        }
        case IS_VISIBLE -> {
            code = String.format("await expect(%s)%s.toBeVisible();",
                    elementVariableName, negativeSection);
        }
        }
        CodeSnippet codeSnippet = new CodeSnippet();
        codeSnippet.setImports(Set.of(PlaywrightNodeImports.EXPECT));
        codeSnippet.setCode(code);
        return codeSnippet;
    }

    private static String getArray(String[] args) {
        String delimitedArgs = Arrays.stream(args)
                .map(arg -> String.format("'%s'", arg))
                .collect(Collectors.joining(", "));
        return String.format("[%s]", delimitedArgs);
    }
}
