package com.vaadin.uitest.codesnippetgeneration;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.vaadin.uitest.model.codesnippetgeneration.Action;
import com.vaadin.uitest.model.codesnippetgeneration.Assertion;
import com.vaadin.uitest.model.codesnippetgeneration.CodeSnippet;
import com.vaadin.uitest.model.codesnippetgeneration.CodeSnippetGenerator;
import com.vaadin.uitest.model.codesnippetgeneration.ElementProperty;
import com.vaadin.uitest.model.codesnippetgeneration.GherkinElement;
import com.vaadin.uitest.model.codesnippetgeneration.LocatedGherkinElement;
import com.vaadin.uitest.model.flow.FlowComponentElement;

public class PlaywrightJavaCodeSnippetGenerator
        implements CodeSnippetGenerator {

    @Override
    public CodeSnippet getLocatorCodeSnippet(GherkinElement gherkinElement,
            String elementVarName, String parentVarName) {
        Set<String> imports = new HashSet<>();
        imports.add(PlaywrightJavaImports.LOCATOR);
        String locatorCode = "";
        Map<ElementProperty, String> identifiers = gherkinElement
                .getIdentifiers();
        if (identifiers.containsKey(ElementProperty.ROLE)) {
            imports.add(PlaywrightJavaImports.ARIA_ROLE);
            String ariaRole = getAriaRole(
                    identifiers.get(ElementProperty.ROLE));
            String ariaRoleStr = "AriaRole." + ariaRole;
            if (identifiers.containsKey(ElementProperty.LABEL)) {
                imports.add(PlaywrightJavaImports.PAGE);
                locatorCode = String.format(
                        "Locator %s = %s.getByRole(%s, new Page.GetByRoleOptions().setName(\"%s\"));",
                        elementVarName, parentVarName, ariaRoleStr,
                        identifiers.get(ElementProperty.LABEL));
            } else {
                locatorCode = String.format("Locator %s = %s.getByRole(%s);",
                        elementVarName, parentVarName, ariaRoleStr);
            }
        } else if (identifiers.containsKey(ElementProperty.LABEL)) {
            if (FlowComponentElement.BUTTON
                    .equals(gherkinElement.getFlowComponentElement())) {
                imports.add(PlaywrightJavaImports.ARIA_ROLE);
                locatorCode = String.format(
                        "Locator %s = %s.getByRole(AriaRole.BUTTON, new Page.GetByRoleOptions().setName(\"%s\"));",
                        elementVarName, parentVarName,
                        identifiers.get(ElementProperty.LABEL));
            } else {
                locatorCode = String.format(
                        "Locator %s = %s.getByLabel(\"%s\");", elementVarName,
                        parentVarName, identifiers.get(ElementProperty.LABEL));
            }
        } else if (identifiers.containsKey(ElementProperty.PLACEHOLDER)) {
            locatorCode = String.format(
                    "Locator %s = %s.getByPlaceholder(\"%s\");", elementVarName,
                    parentVarName,
                    identifiers.get(ElementProperty.PLACEHOLDER));
        } else if (identifiers.containsKey(ElementProperty.ALT_TEXT)) {
            locatorCode = String.format("Locator %s = %s.getByAltText(\"%s\");",
                    elementVarName, parentVarName,
                    identifiers.get(ElementProperty.ALT_TEXT));
        } else if (identifiers.containsKey(ElementProperty.TITLE)) {
            locatorCode = String.format("Locator %s = %s.getByTitle(\"%s\");",
                    elementVarName, parentVarName,
                    identifiers.get(ElementProperty.TITLE));
        } else if (identifiers.containsKey(ElementProperty.TAG_NAME)) {
            locatorCode = String.format("Locator %s = %s.locator(\"%s\");",
                    elementVarName, parentVarName,
                    identifiers.get(ElementProperty.TAG_NAME));
        } else if (identifiers.containsKey(ElementProperty.CSS)) {
            locatorCode = String.format("Locator %s = %s.locator(\"%s\");",
                    elementVarName, parentVarName,
                    identifiers.get(ElementProperty.CSS));
        } else if (identifiers.containsKey(ElementProperty.XPATH)) {
            locatorCode = String.format("Locator %s = %s.locator(\"//%s\");",
                    elementVarName, parentVarName,
                    identifiers.get(ElementProperty.XPATH));
        }
        if (identifiers.containsKey(ElementProperty.TEXT)) {
            if (locatorCode.isEmpty()) {
                locatorCode = String.format(
                        "Locator %s = %s.getByText(\"%s\");", elementVarName,
                        parentVarName, identifiers.get(ElementProperty.TEXT));
            } else {
                locatorCode += String.format(
                        "\n%s = %s.filter(new Locator.FilterOptions().setHasText(\"%s\"));",
                        elementVarName, elementVarName,
                        identifiers.get(ElementProperty.TEXT));
            }
        }
        if (!locatorCode.isEmpty()
                && identifiers.containsKey(ElementProperty.NOT_TEXT)) {
            locatorCode += String.format(
                    "\n%s = %s.filter(new Locator.FilterOptions().setHasNotText(\"%s\"));",
                    elementVarName, elementVarName,
                    identifiers.get(ElementProperty.NOT_TEXT));
        }
        CodeSnippet locatorCodeSnippet = new CodeSnippet();
        locatorCodeSnippet.setCode(locatorCode);
        locatorCodeSnippet.setImports(imports);
        return locatorCodeSnippet;
    }

    @Override
    public CodeSnippet getActionCodeSnippet(LocatedGherkinElement element,
            Action action, String... args) {
        Set<String> imports = new HashSet<>();
        String actionCode = null;
        switch (action) {
        case FILL -> actionCode = String.format("%s.fill(\"%s\");",
                element.getVariableName(), args[0]);
        case CHECK -> actionCode = String.format("%s.check();",
                element.getVariableName());
        case UNCHECK -> actionCode = String.format("%s.uncheck();",
                element.getVariableName());
        case SELECT_OPTION -> {
            if (FlowComponentElement.COMBO_BOX.equals(
                    element.getGherkinElement().getFlowComponentElement())) {
                actionCode = String.format(
                        "%s.click();\npage.getByRole(AriaRole.OPTION, new Page.GetByRoleOptions().setName(\"%s\")).click();",
                        element.getVariableName(), args[0]);
            } else {
                String optionsStr;
                if (args.length == 1) {
                    optionsStr = "\"" + args[0] + "\"";
                } else {
                    String delimitedOptions = Arrays.stream(args)
                            .map(arg -> String.format("\"%s\"", arg))
                            .collect(Collectors.joining(", "));
                    optionsStr = String.format("new String[] {%s}",
                            delimitedOptions);
                }
                actionCode = String.format("%s.selectOption(%s);",
                        element.getVariableName(), optionsStr);
            }
        }
        case CLICK -> actionCode = String.format("%s.click();",
                element.getVariableName());
        case DOUBLE_CLICK -> actionCode = String.format("%s.dblclick();",
                element.getVariableName());
        case PRESS_SEQUENTIALLY ->
            actionCode = String.format("%s.pressSequentially(\"%s\");",
                    element.getVariableName(), args[0]);
        case PRESS -> actionCode = String.format("%s.press(\"%s\");",
                element.getVariableName(), args[0]);
        case SELECT_TEXT -> actionCode = String.format("%s.selectText();",
                element.getVariableName());
        }
        CodeSnippet codeSnippet = new CodeSnippet();
        codeSnippet.setImports(imports);
        codeSnippet.setCode(actionCode);
        return codeSnippet;
    }

    @Override
    public CodeSnippet getAssertionCodeSnippet(LocatedGherkinElement element,
            Assertion assertion, boolean negative, String... args) {
        String code = null;
        String elementVariableName = element.getVariableName();
        String negativeSection = negative ? ".not()" : "";
        switch (assertion) {
        case CONTAINS_TEXT -> {
            code = String.format(
                    "PlaywrightAssertions.assertThat(%s)%s.containsText(\"%s\");",
                    elementVariableName, negativeSection, args[0]);
        }
        case HAS_COUNT -> {
            code = String.format(
                    "PlaywrightAssertions.assertThat(%s)%s.hasCount(%s);",
                    elementVariableName, negativeSection, args[0]);
        }
        case HAS_TEXT -> {
            code = String.format(
                    "PlaywrightAssertions.assertThat(%s)%s.hasText(\"%s\");",
                    elementVariableName, negativeSection, args[0]);
        }
        case HAS_VALUE -> {
            code = String.format(
                    "PlaywrightAssertions.assertThat(%s)%s.hasValue(\"%s\");",
                    elementVariableName, negativeSection, args[0]);
        }
        case HAS_VALUES -> {
            String delimitedArgs = Arrays.stream(args)
                    .map(arg -> String.format("\"%s\"", arg))
                    .collect(Collectors.joining(", "));
            String valuesArray = String.format("new String[] {%s}",
                    delimitedArgs);
            code = String.format(
                    "PlaywrightAssertions.assertThat(%s)%s.hasValues(%s);",
                    elementVariableName, negativeSection, valuesArray);
        }
        case IS_CHECKED -> {
            code = String.format(
                    "PlaywrightAssertions.assertThat(%s)%s.isChecked();",
                    elementVariableName, negativeSection);
        }
        case IS_DISABLED -> {
            code = String.format(
                    "PlaywrightAssertions.assertThat(%s)%s.isDisabled();",
                    elementVariableName, negativeSection);
        }
        case IS_EDITABLE -> {
            code = String.format(
                    "PlaywrightAssertions.assertThat(%s)%s.isEditable();",
                    elementVariableName, negativeSection);
        }
        case IS_EMPTY -> {
            code = String.format(
                    "PlaywrightAssertions.assertThat(%s)%s.isEmpty();",
                    elementVariableName, negativeSection);
        }
        case IS_ENABLED -> {
            code = String.format(
                    "PlaywrightAssertions.assertThat(%s)%s.isEnabled();",
                    elementVariableName, negativeSection);
        }
        case IS_FOCUSED -> {
            code = String.format(
                    "PlaywrightAssertions.assertThat(%s)%s.isFocused();",
                    elementVariableName, negativeSection);
        }
        case IS_HIDDEN -> {
            code = String.format(
                    "PlaywrightAssertions.assertThat(%s)%s.isHidden();",
                    elementVariableName, negativeSection);
        }
        case IS_IN_VIEWPORT -> {
            code = String.format(
                    "PlaywrightAssertions.assertThat(%s)%s.isInViewport();",
                    elementVariableName, negativeSection);
        }
        case IS_VISIBLE -> {
            code = String.format(
                    "PlaywrightAssertions.assertThat(%s)%s.isVisible();",
                    elementVariableName, negativeSection);
        }
        }
        CodeSnippet codeSnippet = new CodeSnippet();
        codeSnippet.setImports(
                Set.of(PlaywrightJavaImports.PLAYWRIGHT_ASSERTIONS));
        codeSnippet.setCode(code);
        return codeSnippet;
    }

    private static String getAriaRole(String role) {
        String cleanedRole = role.replace("_", "").replace("-", "")
                .replace(" ", "").toUpperCase();
        return cleanedRole;
    }
}
