package com.vaadin.uitest.codesnippetgeneration;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import com.vaadin.uitest.model.TestFramework;
import com.vaadin.uitest.model.codesnippetgeneration.Action;
import com.vaadin.uitest.model.codesnippetgeneration.Assertion;
import com.vaadin.uitest.model.codesnippetgeneration.CodeSnippet;
import com.vaadin.uitest.model.codesnippetgeneration.CodeSnippetGenerationArguments;
import com.vaadin.uitest.model.codesnippetgeneration.CodeSnippetGenerator;
import com.vaadin.uitest.model.codesnippetgeneration.GherkinElement;
import com.vaadin.uitest.model.codesnippetgeneration.InteractionArguments;
import com.vaadin.uitest.model.codesnippetgeneration.LocatedGherkinElement;
import com.vaadin.uitest.model.scenario.TestScenarioStep;

public class CodeSnippetProvider {

    private static final TestFramework DEFAULT_TEST_FRAMEWORK = TestFramework.PLAYWRIGHT_JAVA;

    public static CodeSnippet getCodeSnippet(TestScenarioStep gherkinStep) {
        return getCodeSnippet(gherkinStep, DEFAULT_TEST_FRAMEWORK);
    }

    public static CodeSnippet getCodeSnippet(TestScenarioStep gherkinStep,
            TestFramework testFramework) {
        CodeSnippetGenerationArguments codeSnippetGenerationArguments = CodeSnippetGenerationArguments
                .fromGherkinStep(gherkinStep);
        codeSnippetGenerationArguments.setTestFramework(testFramework);
        return getCodeSnippet(codeSnippetGenerationArguments);
    }

    public static CodeSnippet getCodeSnippet(
            CodeSnippetGenerationArguments codeSnippetGenerationArguments) {
        if (codeSnippetGenerationArguments == null
                || codeSnippetGenerationArguments
                        .getElementsInHierarchicalOrder() == null
                || codeSnippetGenerationArguments
                        .getElementsInHierarchicalOrder().isEmpty()
                || codeSnippetGenerationArguments
                        .getInteractionArguments() == null) {
            return null;
        }
        CodeSnippetGenerator codeSnippetGenerator = getCodeSnippetGenerator(
                codeSnippetGenerationArguments.getTestFramework());
        List<LocatedGherkinElement> locatedGherkinElements = locateGherkinElements(
                codeSnippetGenerator, codeSnippetGenerationArguments
                        .getElementsInHierarchicalOrder());
        if (locatedGherkinElements == null) {
            return null;
        }
        LocatedGherkinElement elementToInteractWith = locatedGherkinElements
                .get(locatedGherkinElements.size() - 1);
        CodeSnippet interactionCodeSnippet = getInteractionCodeSnippet(
                codeSnippetGenerator, elementToInteractWith,
                codeSnippetGenerationArguments.getInteractionArguments());
        CodeSnippet finalCodeSnippet = new CodeSnippet();
        finalCodeSnippet.setDescription(
                codeSnippetGenerationArguments.getOriginalGherkinStep());
        finalCodeSnippet.setCode(getFinalCodeSnippet(locatedGherkinElements,
                interactionCodeSnippet));
        finalCodeSnippet.setImports(getFinalImports(locatedGherkinElements,
                interactionCodeSnippet));
        return finalCodeSnippet;
    }

    private static Set<String> getFinalImports(
            List<LocatedGherkinElement> locatedGherkinElements,
            CodeSnippet interactionCodeSnippet) {
        Set<String> imports = new HashSet<>();
        locatedGherkinElements.stream()
                .map(LocatedGherkinElement::getLocatorCodeSnippet)
                .map(CodeSnippet::getImports).filter(Objects::nonNull)
                .forEach(imports::addAll);
        if (interactionCodeSnippet.getImports() != null) {
            imports.addAll(interactionCodeSnippet.getImports());
        }
        return imports;
    }

    private static String getFinalCodeSnippet(
            List<LocatedGherkinElement> locatedGherkinElements,
            CodeSnippet interactionCodeSnippet) {
        List<CodeSnippet> codeSnippets = new ArrayList<>();
        locatedGherkinElements.stream()
                .map(LocatedGherkinElement::getLocatorCodeSnippet)
                .forEach(codeSnippets::add);
        codeSnippets.add(interactionCodeSnippet);
        return codeSnippets.stream().map(CodeSnippet::getCode)
                .collect(Collectors.joining("\n"));
    }

    private static CodeSnippet getInteractionCodeSnippet(
            CodeSnippetGenerator codeSnippetGenerator,
            LocatedGherkinElement element,
            InteractionArguments interactionArguments) {
        CodeSnippet interactionCode = null;
        if (interactionArguments.getInteraction() instanceof Action action) {
            interactionCode = codeSnippetGenerator.getActionCodeSnippet(element,
                    action, interactionArguments.getArgs());
        } else if (interactionArguments
                .getInteraction() instanceof Assertion assertion) {
            interactionCode = codeSnippetGenerator.getAssertionCodeSnippet(
                    element, assertion, interactionArguments.isNegative(),
                    interactionArguments.getArgs());
        }
        return interactionCode;
    }

    private static List<LocatedGherkinElement> locateGherkinElements(
            CodeSnippetGenerator codeSnippetGenerator,
            List<GherkinElement> elementsInHierarchicalOrder) {
        String baseElementName = "page";
        List<LocatedGherkinElement> locatedGherkinElements = new ArrayList<>();
        for (int i = 0; i < elementsInHierarchicalOrder.size(); i++) {
            LocatedGherkinElement locatedGherkinElement;
            GherkinElement gherkinElement = elementsInHierarchicalOrder.get(i);
            String elementVarName = enumToCamelcase(
                    gherkinElement.getFlowComponentElement().name());
            if (i == 0) {
                locatedGherkinElement = locateGherkinElement(
                        codeSnippetGenerator, gherkinElement, elementVarName,
                        baseElementName);
            } else {
                long numberOfSameElements = locatedGherkinElements.stream()
                        .map(LocatedGherkinElement::getGherkinElement)
                        .map(GherkinElement::getFlowComponentElement)
                        .filter(gherkinElement
                                .getFlowComponentElement()::equals)
                        .count();
                if (numberOfSameElements > 0) {
                    elementVarName += (numberOfSameElements + 1);
                }
                locatedGherkinElement = locateGherkinElement(
                        codeSnippetGenerator, gherkinElement, elementVarName,
                        locatedGherkinElements.get(i - 1).getVariableName());
            }
            if (locatedGherkinElement == null) {
                return null;
            }
            locatedGherkinElements.add(locatedGherkinElement);
        }
        return locatedGherkinElements;
    }

    private static LocatedGherkinElement locateGherkinElement(
            CodeSnippetGenerator codeSnippetGenerator,
            GherkinElement gherkinElement, String elementVarName,
            String parentVarName) {
        CodeSnippet locatorCodeSnippet = codeSnippetGenerator
                .getLocatorCodeSnippet(gherkinElement, elementVarName,
                        parentVarName);
        if (locatorCodeSnippet.getCode().isEmpty()) {
            return null;
        }
        LocatedGherkinElement locatedGherkinElement = new LocatedGherkinElement();
        locatedGherkinElement.setGherkinElement(gherkinElement);
        locatedGherkinElement.setLocatorCodeSnippet(locatorCodeSnippet);
        locatedGherkinElement.setVariableName(elementVarName);
        return locatedGherkinElement;
    }

    private static CodeSnippetGenerator getCodeSnippetGenerator(
            TestFramework testFramework) {
        switch (testFramework) {
        case PLAYWRIGHT_JAVA -> {
            return new PlaywrightJavaCodeSnippetGenerator();
        }
        case PLAYWRIGHT_NODE -> {
            return new PlaywrightNodeCodeSnippetGenerator();
        }
        }
        return null;
    }

    public static String enumToCamelcase(String name) {
        final String[] parts = name.toLowerCase().split("_");
        final StringBuilder sb = new StringBuilder();
        for (int i = 0; i < parts.length; i++) {
            String s = parts[i];
            if (i == 0) {
                sb.append(s);
            } else if (!s.isEmpty()) {
                String c = s.substring(0, 1);
                sb.append(s.replaceFirst(c, c.toUpperCase()));
            }
        }
        return new String(sb);
    }
}
