package com.vaadin.uitest.browser;

import static com.vaadin.uitest.browser.ChromeBrowser.DATA_FOLDER;
import static com.vaadin.uitest.browser.ChromeBrowser.WAIT_FOR_VAADIN_SCRIPT;

import java.io.IOException;
import java.time.Duration;
import java.util.Calendar;
import java.util.Date;

import com.vaadin.uitest.generator.utils.CopyUtils;

import org.openqa.selenium.By;
import org.openqa.selenium.Cookie;
import org.openqa.selenium.Dimension;
import org.openqa.selenium.WebElement;
import org.openqa.selenium.chrome.ChromeDriver;
import org.openqa.selenium.chrome.ChromeOptions;
import org.openqa.selenium.support.ui.ExpectedConditions;
import org.openqa.selenium.support.ui.WebDriverWait;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ChromeLogin implements BrowserLogin {

    private static final Logger LOGGER = LoggerFactory
            .getLogger(ChromeLogin.class);

    private String url;
    private String loginSelector;
    private String passwordSelector;
    private String buttonSelector;
    private String login;
    private String password;
    private ChromeDriver driver;

    @Override
    public BrowserLogin setParameters(String url, String loginSelector,
            String passwordSelector, String buttonSelector, String login,
            String password) {
        this.url = url;
        this.loginSelector = loginSelector;
        this.passwordSelector = passwordSelector;
        this.buttonSelector = buttonSelector;
        this.login = login;
        this.password = password;
        return this;
    }

    public String doLogin() throws InterruptedException, IOException {
        ChromeOptions options = new ChromeOptions();

        options.addArguments("--no-sandbox", "--test-mode",
                "--disable-search-engine-choice-screen");

        // configure the browser data-folder to save cookie
        if (DATA_FOLDER.exists()) {
            CopyUtils.deleteDirectory(DATA_FOLDER.toPath());
        }
        DATA_FOLDER.mkdirs();
        Runtime.getRuntime().addShutdownHook(new Thread(() -> {
            try {
                CopyUtils.deleteDirectory(DATA_FOLDER.toPath());
            } catch (IOException ignored) {
            }
        }));
        options.addArguments("user-data-dir=" + DATA_FOLDER.getPath());

        // login process is headless only in the case that we have
        // enough parameters for doing automatic login
        boolean headless = (System.getProperty("headless") == null
                || Boolean.getBoolean("headless"))
                && !(System.getProperty("headed") != null
                        && (System.getProperty("headed").isEmpty() || Boolean
                                .parseBoolean(System.getProperty("headed"))));
        headless = headless && login != null && password != null;
        if (headless) {
            options.addArguments("--headless");
        }

        // open the browser and visit login page
        driver = new ChromeDriver(options);
        driver.manage().window().setSize(new Dimension(1024, 800));
        driver.manage().timeouts().pageLoadTimeout(Duration.ofSeconds(30));
        driver.manage().timeouts().scriptTimeout(Duration.ofSeconds(30));
        driver.manage().timeouts().implicitlyWait(Duration.ofSeconds(30));
        driver.get(url);

        WebDriverWait wait = new WebDriverWait(driver, Duration.ofSeconds(60));
        wait.until(ExpectedConditions.jsReturnsValue(WAIT_FOR_VAADIN_SCRIPT));

        // save the html before the login process
        String loginHtml = (String) driver
                .executeScript("return document.body.outerHTML");

        // if we have enough parameters, perform automatic login
        if (login != null && loginSelector != null) {
            WebElement loginElm = driver
                    .findElement(By.cssSelector(loginSelector));
            if (loginElm != null) {
                loginElm.sendKeys(login);
                if (password != null && passwordSelector != null) {
                    WebElement passElm = driver
                            .findElement(By.cssSelector(passwordSelector));
                    if (passElm != null) {
                        passElm.sendKeys(password);
                        if (!password.endsWith("\n")
                                && buttonSelector != null) {
                            WebElement buttonElm = driver.findElement(
                                    By.cssSelector(buttonSelector));
                            if (buttonElm != null) {
                                buttonElm.click();
                            }
                        }
                    }
                }
            }
        }

        // wait until URL changes in the app
        wait.until(ExpectedConditions.not(ExpectedConditions.urlToBe(url)));

        // extra sleep for being sure login process finishes
        Thread.sleep(500);

        // when the session cookie does not have expiration is not persisted
        // thus, we replace cookies without date with expiration of some minutes
        for (Cookie c : driver.manage().getCookies()) {
            if (c.getExpiry() == null) {
                Calendar cal = Calendar.getInstance();
                cal.add(Calendar.MINUTE, 10);
                Date date = cal.getTime();
                driver.manage().deleteCookie(c);
                c = new Cookie(c.getName(), c.getValue(), c.getDomain(),
                        c.getPath(), date, c.isSecure(), c.isHttpOnly(),
                        c.getSameSite());
                driver.manage().addCookie(c);
            }
        }
        return loginHtml;
    }

    public void quit() {
        driver.close();
    }

    public void clean() {
        if (DATA_FOLDER.exists()) {
            try {
                CopyUtils.deleteDirectory(DATA_FOLDER.toPath());
            } catch (IOException e) {
                LOGGER.error(e.getMessage(), e);
            }
        }
    }

}