package com.vaadin.uitest.browser;

import java.io.File;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.time.Duration;
import java.util.Collections;
import java.util.List;

import com.vaadin.uitest.generator.utils.CopyUtils;

import org.openqa.selenium.Dimension;
import org.openqa.selenium.chrome.ChromeDriver;
import org.openqa.selenium.chrome.ChromeOptions;
import org.openqa.selenium.support.ui.ExpectedConditions;
import org.openqa.selenium.support.ui.WebDriverWait;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ChromeBrowser implements Browser {

    private static final Logger LOGGER = LoggerFactory
            .getLogger(ChromeBrowser.class);

    private static final boolean headless = !ManagementFactory
            .getRuntimeMXBean().getInputArguments().toString().contains("jdwp")
            && (System.getProperty("headless") == null
                    || Boolean.getBoolean("headless"))
            && !(System.getProperty("headed") != null
                    && (System.getProperty("headed").isEmpty() || Boolean
                            .parseBoolean(System.getProperty("headed"))));

    // This block has been borrowed from TestBench
    static String WAIT_FOR_VAADIN_SCRIPT =
    // @formatter:off
            "if (window.Vaadin && window.Vaadin.Flow && window.Vaadin.Flow.clients) {"
            + "  var clients = window.Vaadin.Flow.clients;"
            + "  for (var client in clients) {"
            + "    if (clients[client].isActive()) { "
            + "      return false;"
            + "    }"
            + "  }"
            + "  return true;"
            + "} else if (window.Vaadin && window.Vaadin.Flow && window.Vaadin.Flow.devServerIsNotLoaded) {"
            + "  return false;"
            + "} else {"
            + "  return true;"
            + "}";
    // @formatter:on

    public static File DATA_FOLDER = new File(
            new File(System.getProperty("java.io.tmpdir")),
            "ui-test-generator-data-folder");

    private static int count = 0;
    private final ChromeDriver driver;
    private final File data;

    private static synchronized int increaseCounter() {
        return count++;
    }

    public ChromeBrowser() {
        ChromeOptions options = new ChromeOptions();
        options.addArguments("--no-sandbox", "--test-mode",
                "--disable-search-engine-choice-screen");
        if (headless) {
            options.addArguments("--headless");
        }
        // when there is a valid data-folder use it so as we have the
        // login credentials. But first we make a copy so as parallelism
        // works, and we don't modify cookies in the case of logout
        if (DATA_FOLDER.exists()) {
            data = new File(DATA_FOLDER.getAbsolutePath() + increaseCounter());
            try {
                options.addArguments("user-data-dir=" + data.getPath());
                if (data.exists()) {
                    CopyUtils.deleteDirectory(data.toPath());
                }
                copyDirectory(DATA_FOLDER.toPath(), data.toPath());
            } catch (IOException e) {
                LOGGER.error(e.getMessage(), e);
            }
        } else {
            data = null;
        }
        driver = new ChromeDriver(options);
        driver.manage().window().setSize(new Dimension(1024, 800));
        driver.manage().timeouts().pageLoadTimeout(Duration.ofSeconds(5));
        driver.manage().timeouts().scriptTimeout(Duration.ofSeconds(2));
        driver.manage().timeouts().implicitlyWait(Duration.ofSeconds(2));
    }

    public String getHTMLContent(String url) {
        return getHTMLContent(url, Collections.emptyList());
    }

    public String getHTMLContent(String url, List<String> jsUiActions) {
        goToUrl(url);
        jsUiActions.forEach(driver::executeScript);
        return (String) driver.executeScript("return document.body.innerHTML");
    }

    public String getHTMLContent(String url, String selector) {
        goToUrl(url);
        // capture the DOM structure defined by the css selector
        return (String) driver
                .executeScript("return document.body"
                        + ((selector == null || "body".equals(selector)) ? ""
                                : ".querySelector('" + selector + "')")
                        + ".outerHTML");
    }

    private void goToUrl(String url) {
        driver.get(url);
        WebDriverWait wait = new WebDriverWait(driver, Duration.ofSeconds(60));
        wait.until(ExpectedConditions.jsReturnsValue(WAIT_FOR_VAADIN_SCRIPT));
        // do an extra sleep after to assure that async process finished
        try {
            Thread.sleep(500);
        } catch (InterruptedException e) {
        }
    }

    public void close() {
        driver.quit();
        if (data != null && data.exists()) {
            try {
                CopyUtils.deleteDirectory(data.toPath());
            } catch (IOException e) {
                LOGGER.error(e.getMessage(), e);
            }
        }
    }

    private static void copyDirectory(Path source, Path target)
            throws IOException {
        Files.walkFileTree(source, new SimpleFileVisitor<>() {
            @Override
            public FileVisitResult preVisitDirectory(Path dir,
                    BasicFileAttributes attrs) throws IOException {
                Files.createDirectories(target.resolve(source.relativize(dir)));
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFile(Path file,
                    BasicFileAttributes attrs) throws IOException {
                Files.copy(file, target.resolve(source.relativize(file)),
                        StandardCopyOption.REPLACE_EXISTING);
                return FileVisitResult.CONTINUE;
            }
        });
    }

    public static Boolean isServerRunning(String url) {
        ChromeBrowser browser = new ChromeBrowser();
        try {
            String content = browser.getHTMLContent(url);
            return content != null && !content.isEmpty();
        } catch (Exception e) {
            return false;
        } finally {
            browser.close();
        }
    }
}