package com.vaadin.uitest.ai.services.assistant;

import static com.vaadin.uitest.ai.services.AiServiceConstants.MODEL;
import static com.vaadin.uitest.ai.services.AiServiceConstants.OPENAI_API_COMPLETIONS;
import static com.vaadin.uitest.ai.services.AiServiceConstants.OPENAI_API_URL;
import static com.vaadin.uitest.ai.services.AiServiceConstants.TEMPERATURE;

import java.io.IOException;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Map;
import java.util.Scanner;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.vaadin.uitest.ai.utils.KeysUtils;
import com.vaadin.uitest.model.chat.ChatCompletionChunkResponse;
import com.vaadin.uitest.model.chat.ChatCompletionMessage;
import com.vaadin.uitest.model.chat.ChatCompletionMessageIn;

public class AssistantService {
    private static final Logger LOGGER = LoggerFactory
            .getLogger(AssistantService.class);

    private static final ObjectMapper MAPPER = new ObjectMapper();

    public AssistantService() {
    }

    /**
     * Calls the OpenAI chat completion API.
     *
     * @return The completion as a stream of chunks
     */
    public String getCompletionStream(List<ChatCompletionMessage> history,
            String sessionId) {
        LOGGER.debug("History size {}", history.size());
        if (history.isEmpty()) {
            LOGGER.error("History is empty");
            return "ChatCompletionHistory was empty. Could not generate completion.";
        }
        return generateCompletionStringForAssistant(history);
    }

    /**
     * Calls the OpenAI chat completion API synchronously.
     *
     * @return The completion as a String with the content
     */
    public String getCompletionString(List<ChatCompletionMessage> history) {
        LOGGER.debug("History size {}", history.size());
        if (history.isEmpty()) {
            throw new RuntimeException("History is empty");
        }
        return generateCompletionString(history);
    }

    public String generateCompletionStringForAssistant(
            List<ChatCompletionMessage> messages) {
        if (Boolean.getBoolean("ai.debug")) {
            LOGGER.info("Generating completion for messages: {}", messages);
        }

        Map<String, Object> map = Map.of("model", MODEL, "messages",
                messages.stream()
                        .map(msg -> new ChatCompletionMessageIn(
                                ChatCompletionMessageIn.Role
                                        .valueOf(msg.getRole().name()),
                                msg.getContent()))
                        .collect(Collectors.toList()),
                "temperature", TEMPERATURE);
        return sendRequest(map);
    }

    public String generateCompletionString(
            List<ChatCompletionMessage> history) {
        Map<String, Object> map = Map.of("model", MODEL, "messages", history
                .stream().map(
                        msg -> new ChatCompletionMessageIn(
                                ChatCompletionMessageIn.Role
                                        .valueOf(msg.getRole().name()),
                                msg.getContent()))
                .collect(Collectors.toList()));
        return sendRequest(map);
    }

    private String sendRequest(Map<String, Object> map) {
        try {
            String requestJson = MAPPER.writeValueAsString(map);

            URL url = new URL(OPENAI_API_URL + OPENAI_API_COMPLETIONS);

            HttpURLConnection con = (HttpURLConnection) url.openConnection();
            con.setRequestMethod("POST");
            con.setRequestProperty("Content-Type", "application/json");
            con.setRequestProperty("Authorization",
                    "Bearer " + KeysUtils.getOpenAiKey());
            con.setDoOutput(true);
            con.setConnectTimeout(2000);
            con.setReadTimeout(60000);

            try (OutputStream os = con.getOutputStream()) {
                byte[] input = requestJson.getBytes(StandardCharsets.UTF_8);
                os.write(input, 0, input.length);
            }
            try (Scanner scanner = new Scanner(con.getInputStream(),
                    StandardCharsets.UTF_8)) {
                String responseBody = scanner.useDelimiter("\\A").next();
                LOGGER.debug(responseBody);
                return MAPPER
                        .readValue(responseBody,
                                ChatCompletionChunkResponse.class)
                        .getChoices().get(0).getDelta().getContent();
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

}
