/*
 * Copyright 2000-2025 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.pro.licensechecker;

import java.util.Objects;

/**
 * Exception thrown when a pre-trial validation fails due to specific conditions
 * of the pre-trial state. This class extends {@code LicenseValidationException}
 * to provide detailed contextual information about the pre-trial state that
 * caused the exception.
 */
public class PreTrialLicenseValidationException
        extends LicenseValidationException {
    private final PreTrial preTrial;

    /**
     * Constructs a new {@code PreTrialLicenseValidationException} with a given
     * pre-trial state. The exception message is determined based on the state
     * of the provided {@code PreTrial} object.
     *
     * @param preTrial
     *            the {@code PreTrial} instance containing details about the
     *            pre-trial state. Must not be {@code null}. Throws a
     *            {@code NullPointerException} if {@code preTrial} is null.
     */
    public PreTrialLicenseValidationException(PreTrial preTrial) {
        super(computeMessage(
                Objects.requireNonNull(preTrial, "pre trial cannot be null")));
        this.preTrial = preTrial;
    }

    /**
     * Retrieves the {@code PreTrial} instance associated with this exception.
     * <p>
     * Information can be used to build detailed messages about the pre-trial
     * state for the end-user.
     *
     * @return the {@code PreTrial} instance containing details about the
     *         pre-trial state that caused the exception
     */
    public PreTrial getPreTrial() {
        return preTrial;
    }

    private static String computeMessage(PreTrial preTrial) {
        String message;
        switch (preTrial.trialState) {
        case START_ALLOWED:
            message = "Pre trial not started yet.";
            break;
        case ACCESS_DENIED:
            message = "Pre trial does not grant access to the product.";
            break;
        case EXPIRED:
            message = "Pre trial period has expired. A new pre trial can be started in "
                    + preTrial.daysRemainingUntilRenewal + " days.";
            break;
        case RUNNING:
            message = "Pre trial is active. It will expire in "
                    + preTrial.daysRemaining + " days.";
            break;
        default:
            message = "Pre trial state undefined.";
        }
        return message;
    }

}
