/*
 * Copyright 2000-2024 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.pro.licensechecker;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

import org.slf4j.Logger;

/**
 * Fetches an offline key from either the home directory (~/.vaadin/offlineKey)
 * or from a system property ({@code vaadin.offlineKey}) if available. Also
 * capable of writing an offline key but this should not be used except for
 * testing. Offline keys are always downloaded and stored manually.
 */
public class LocalOfflineKey {

    private static OfflineKey read(File offlineKeyLocation) throws IOException {
        if (!offlineKeyLocation.exists()) {
            return null;
        }
        try (FileInputStream is = new FileInputStream(offlineKeyLocation)) {
            OfflineKey key = new OfflineKey(Util.toString(is),
                    OfflineKeySource.FILE,
                    offlineKeyLocation.getAbsolutePath());

            // Validate machine ID format matches the file name
            String fileName = offlineKeyLocation.getName();
            String machineId = key.getMachineId();

            if (machineId != null) {
                if (fileName.equals("offlineKeyV2")
                        || fileName.equals("offlineKeyV2.txt")) {
                    // offlineKeyV2 must contain mid2- prefix
                    if (!machineId.startsWith("mid2-")) {
                        getLogger().error(
                                "Invalid machine ID format in {}: expected 'mid2-*' but got '{}'",
                                fileName, machineId);
                        return null;
                    }
                } else if (fileName.equals("offlineKey")
                        || fileName.equals("offlineKey.txt")) {
                    // offlineKey must contain mid- prefix (or legacy test ID
                    // aa1122)
                    if (!machineId.startsWith("mid-")
                            && !machineId.equals("aa1122")) {
                        getLogger().error(
                                "Invalid machine ID format in {}: expected 'mid-*' but got '{}'",
                                fileName, machineId);
                        return null;
                    }
                }
            }

            return key;
        } catch (ParseException e) {
            getLogger().error("Unable to read offline license from file", e);
            return null;
        }
    }

    /**
     * Returns all file locations where offline keys are searched, in priority
     * order.
     *
     * @return array of files that can contain offline keys
     */
    public static File[] getLocations() {
        return Stream
                .of("offlineKeyV2", "offlineKeyV2.txt", "offlineKey",
                        "offlineKey.txt", "serverKey", "serverKey.txt")
                .map(name -> new File(LocalProKey.getVaadinFolder(), name))
                .toArray(File[]::new);
    }

    /**
     * Finds an offline key, either from a system property or from the home
     * directory.
     *
     * @return an offline key or {@code null} if no key was found
     */
    public static OfflineKey get() {
        List<OfflineKey> allKeys = getAll();

        if (allKeys.isEmpty()) {
            return null;
        }

        OfflineKey first = allKeys.get(0);

        // Log the source to aid debugging of license issues
        if (first.getSource() != null) {
            switch (first.getSource()) {
            case SYSTEM_PROPERTY:
                getLogger().debug("Using offlineKey from system property");
                break;
            case ENVIRONMENT_VARIABLE:
                getLogger()
                        .debug("Using offlineKey from environment variable");
                break;
            case FILE:
                getLogger().debug("Found offline key in {}",
                        first.getSourceDetail());
                break;
            }
        }

        return first;
    }

    /**
     * Returns all offline keys found from all sources (system properties,
     * environment variables, and file system), in priority order. This allows
     * callers to implement custom selection logic.
     *
     * @return list of OfflineKey objects found from all sources, in priority
     *         order
     */
    public static List<OfflineKey> getAll() {
        List<OfflineKey> keys = new ArrayList<>();

        // Check system property first (highest priority)
        OfflineKey systemPropertyKey = getSystemProperty();
        if (systemPropertyKey != null) {
            keys.add(systemPropertyKey);
        }

        // Check environment variable second
        OfflineKey environmentVariableKey = getEnvironmentVariable();
        if (environmentVariableKey != null) {
            keys.add(environmentVariableKey);
        }

        // Check file locations last (lowest priority)
        try {
            for (File location : getLocations()) {
                OfflineKey key = read(location);
                if (key != null) {
                    keys.add(key);
                }
            }
        } catch (IOException e) {
            getLogger().debug("Unable to read offline keys", e);
        }
        return keys;
    }

    /**
     * Creates a new {@link OfflineKey} based on the given string
     * representation.
     *
     * @param key
     *            The string representation of the offline key.
     * @return an {@link OfflineKey}, or an empty {@link Optional} if the given
     *         string does not represent a valid offline key.
     */
    public static Optional<OfflineKey> fromString(String key) {
        if (key != null) {
            return Optional.ofNullable(parseOfflineKey(key, null));
        }
        return Optional.empty();
    }

    private static OfflineKey getSystemProperty() {
        String jwtData = System.getProperty("vaadin.offlineKey");
        if (jwtData == null) {
            jwtData = System.getProperty("vaadin.key");
            if (jwtData == null) {
                return null;
            }
        }
        return parseOfflineKey(jwtData, OfflineKeySource.SYSTEM_PROPERTY);
    }

    private static OfflineKey getEnvironmentVariable() {
        String value = EnvironmentVariables.get("VAADIN_OFFLINE_KEY");
        if (value == null) {
            value = EnvironmentVariables.get("VAADIN_KEY");
            if (value == null) {
                return null;
            }
        }
        return parseOfflineKey(value,
                OfflineKeySource.ENVIRONMENT_VARIABLE);
    }

    private static OfflineKey parseOfflineKey(String jwtData,
            OfflineKeySource source) {
        try {
            return new OfflineKey(jwtData, source, null);
        } catch (ParseException e) {
            if (source != null) {
                getLogger().error("Unable to read offline license from {}",
                        source, e);
            }
            return null;
        }
    }

    private static Logger getLogger() {
        return LicenseChecker.getLogger();
    }

}
