/*
 * Copyright 2000-2024 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.pro.licensechecker;

import java.math.BigInteger;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.time.Instant;
import java.time.Period;
import java.time.temporal.TemporalAmount;
import java.util.Base64;
import java.util.prefs.Preferences;

import org.slf4j.Logger;

public class History {

    private static final String CHECK_PREFIX = "lastcheck-";
    public static final int DEFAULT_PRE_TRIAL_DURATION_DAYS = 7;

    private static Logger getLogger() {
        return LicenseChecker.getLogger();
    }

    static boolean isRecentlyValidated(Product product, BuildType buildType,
            ProKey proKey) {
        return isRecentlyValidated(product, Period.ofDays(1), buildType,
                proKey);
    }

    static boolean isRecentlyValidated(Product product, TemporalAmount period,
            BuildType buildType, ProKey proKey) {
        getLogger().debug(
                "Checking if license for {} has recently been checked for build type {} and pro key {}",
                product, buildType, proKey);
        Instant lastCheck = getLastCheckTime(product, buildType, proKey);
        return isRecentlyValidated(lastCheck, period);
    }

    private static boolean isRecentlyValidated(Instant lastCheck,
            TemporalAmount period) {
        if (lastCheck == null) {
            return false;
        }
        Instant now = Instant.now();
        if (lastCheck.isAfter(now)) {
            // Invalid last check value
            return false;
        }

        Instant nextCheck = lastCheck.plus(period);
        return now.isBefore(nextCheck);
    }

    public static Instant getLastCheckTime(Product product, BuildType buildType,
            ProKey proKey) {
        String lastCheckKey = getLastCheckKey(product, buildType, proKey);
        return getLastCheckTime(product, lastCheckKey);
    }

    private static Instant getLastCheckTime(Product product,
            String lastCheckKey) {
        long lastCheck = getPreferences().getLong(lastCheckKey, -1);
        if (lastCheck == -1) {
            getLogger().debug("License for {} has never been checked", product);
            return null;
        } else {
            Instant lastCheckInstant = Instant.ofEpochMilli(lastCheck);
            getLogger().debug("Last check for {} was on {}", product,
                    lastCheckInstant);
            return lastCheckInstant;
        }
    }

    public static String getLastSubscription(Product product, ProKey proKey) {
        String lastSubscriptionKey = getLastSubscriptionKey(product, proKey);
        return getPreferences().get(lastSubscriptionKey, "");
    }

    static long setLastCheckTimeNow(Product product, BuildType buildType,
            ProKey proKey) {
        getLogger().debug(
                "Marking license for {} as checked now for buildType {}",
                product, buildType);
        return setLastCheck(product, Instant.now(), buildType, proKey);
    }

    static long setLastCheck(Product product, Instant instant,
            BuildType buildType, ProKey proKey) {
        getLogger().debug(
                "Marking license for {} as checked on {} for buildType {} and pro key {}",
                product, instant, buildType, proKey);

        long timeMillis = instant.toEpochMilli();
        getPreferences().putLong(getLastCheckKey(product, buildType, proKey),
                timeMillis);
        return timeMillis;
    }

    public static String setLastSubscription(Product product,
            String subscription, ProKey proKey) {
        getLogger().debug("Storing subscription for {}", product);

        getPreferences().put(getLastSubscriptionKey(product, proKey),
                subscription);
        return subscription;
    }

    static String getLastCheckKey(Product product, BuildType buildType,
            ProKey proKey) {
        String key = getKey(product, proKey);
        if (buildType != null) {
            key += "-" + buildType.getKey();
        }
        return hash(key);
    }

    static String hash(String key) {
        // We use a hash as there is a max length of 80 characters for the key
        // and we do not really know how long product names and versions will be
        // used
        try {

            MessageDigest md = MessageDigest.getInstance("MD5");
            md.update(key.getBytes());
            byte[] digest = md.digest();
            BigInteger no = new BigInteger(1, digest);
            String hash = no.toString(16);
            while (hash.length() < 32) {
                hash = "0" + hash;
            }

            return CHECK_PREFIX + hash;
        } catch (NoSuchAlgorithmException e) {
            // This does not really happen
            return (CHECK_PREFIX + key).substring(0,
                    Preferences.MAX_KEY_LENGTH);
        }

    }

    private static String getKey(Product product, ProKey proKey) {
        String suffix = proKey == null ? "null" : proKey.getProKey();
        return (product != null
                ? product.getName() + "-" + product.getVersion() + "-"
                : "null-null-") + suffix;
    }

    static String getLastSubscriptionKey(Product product, ProKey proKey) {
        String key = getKey(product, proKey) + "-sub";
        return hash(key);
    }

    private static String getPreTrialKey(Product product, MachineId machineId) {
        return getKey(product, new ProKey(null, machineId.getPrimaryIdFull()));
    }

    static String getLastPreTrialCheckKey(Product product,
            MachineId machineId) {
        String key = getPreTrialKey(product, machineId) + "-pretrial";
        return hash(key);
    }

    static String getLastPreTrialPeriodKey(Product product,
            MachineId machineId) {
        String key = getPreTrialKey(product, machineId) + "-pre-trial-period";
        return hash(key);
    }

    static String getLastDaysRemainingUntilRenewalKey(Product product,
            MachineId machineId) {
        String key = getPreTrialKey(product, machineId)
                + "-pre-trial-days-remaining-until-renewal";
        return hash(key);
    }

    static String getLastNameKey(Product product, MachineId machineId) {
        String key = getPreTrialKey(product, machineId) + "-pre-trial-name";
        return hash(key);
    }

    static String getLastStateKey(Product product, MachineId machineId) {
        String key = getPreTrialKey(product, machineId) + "-pre-trial-state";
        return hash(key);
    }

    private static Preferences getPreferences() {
        return Preferences.userNodeForPackage(OnlineKeyValidator.class);
    }

    static long setLastPreTrialCheckTimeNow(Product product,
            MachineId machineId, PreTrial pretrial) {
        return setLastPreTrialCheckTime(product, machineId, Instant.now(),
                pretrial);
    }

    static long setLastPreTrialCheckTime(Product product, MachineId machineId,
            Instant instant, PreTrial pretrial) {
        if (pretrial.getDaysRemaining() < 0) {
            throw new IllegalArgumentException(
                    "Invalid trial period: " + pretrial.getDaysRemaining());
        }
        getLogger().debug(
                "Marking pre-trial check for {} as checked on {} for machine id {}",
                product, instant, machineId.getPrimaryIdFull());

        long timeMillis = instant.toEpochMilli();
        getPreferences().putLong(getLastPreTrialCheckKey(product, machineId),
                timeMillis);
        String lastPreTrialPeriodKey = getLastPreTrialPeriodKey(product,
                machineId);
        String lastDaysRemainingUntilRenewal = getLastDaysRemainingUntilRenewalKey(
                product, machineId);
        // Bit-shifts the trial period value to make it a bit more difficult to
        // identify and modify in the preferences storage. This helps prevent
        // users from easily extending their trial by manually editing
        // preference values.
        // Perhaps this obfuscation is not necessary.
        getPreferences().putLong(lastPreTrialPeriodKey,
                (long) pretrial.getDaysRemaining() << 32);
        getPreferences().putLong(lastDaysRemainingUntilRenewal,
                (long) pretrial.getDaysRemainingUntilRenewal() << 32);
        // encode name and state to Base64
        getPreferences().put(getLastNameKey(product, machineId),
                Base64.getEncoder()
                        .encodeToString(pretrial.getTrialName().getBytes()));
        getPreferences().put(getLastStateKey(product, machineId),
                Base64.getEncoder().encodeToString(
                        pretrial.getTrialState().name().getBytes()));

        return timeMillis;
    }

    /**
     * @deprecated Use {@link #getLastPreTrialCheckTime(Product, MachineId)}
     *             instead.
     */
    @Deprecated
    public static Instant getLastPreTrialCheckTime(Product product,
            String machineId) {
        return getLastPreTrialCheckTime(product, new MachineId(machineId));
    }

    public static Instant getLastPreTrialCheckTime(Product product,
            MachineId machineId) {
        String lastCheckKey = getLastPreTrialCheckKey(product, machineId);
        return getLastCheckTime(product, lastCheckKey);
    }

    static boolean isPreTrialRecentlyValidated(Product product,
            MachineId machineId) {
        getLogger().debug(
                "Checking if pre-trial for {} has recently been checked for machine id {}",
                product, machineId.getPrimaryIdFull());
        Instant lastCheck = getLastPreTrialCheckTime(product, machineId);
        String lastPeriodKey = getLastPreTrialPeriodKey(product, machineId);
        // Un-shift the stored value to get the pre trial duration
        long preTrialDays = getPreferences().getLong(lastPeriodKey,
                DEFAULT_PRE_TRIAL_DURATION_DAYS) >> 32;
        return isRecentlyValidated(lastCheck,
                Period.ofDays((int) preTrialDays));
    }

    static PreTrial getRecentlyValidatedPreTrial(Product product,
            MachineId machineId) {
        if (!isPreTrialRecentlyValidated(product, machineId)) {
            return null;
        }
        String lastNameKey = getLastNameKey(product, machineId);
        String lastStateKey = getLastStateKey(product, machineId);
        String lastPeriodKey = getLastPreTrialPeriodKey(product, machineId);
        String lastDaysRemainingUntilRenewalKey = getLastDaysRemainingUntilRenewalKey(
                product, machineId);
        if (getPreferences().get(lastStateKey, null) == null) {
            // makes this method backwards compatible as previous versions did
            // not store the state.
            return null;
        }

        // Un-shift the stored value to get the pre trial data
        long preTrialDays = getPreferences().getLong(lastPeriodKey,
                DEFAULT_PRE_TRIAL_DURATION_DAYS) >> 32;
        long daysRemainingUntilRenewal = getPreferences()
                .getLong(lastDaysRemainingUntilRenewalKey, 30) >> 32;
        // decode name and state from Base64
        String name = new String(Base64.getDecoder()
                .decode(getPreferences().get(lastNameKey, "")));
        String state = new String(
                Base64.getDecoder().decode(getPreferences().get(lastStateKey,
                        PreTrial.PreTrialState.START_ALLOWED.name())));
        return new PreTrial(name, PreTrial.PreTrialState.valueOf(state),
                (int) preTrialDays, (int) daysRemainingUntilRenewal);

    }

    /**
     * Clears the local check history so that all products are checked again.
     */
    public static void clearAll() throws Exception {
        Preferences preferences = getPreferences();
        for (String key : preferences.keys()) {
            if (key.startsWith(CHECK_PREFIX)) {
                preferences.remove(key);
            }
        }
    }

}
