/*
 * Copyright 2000-2025 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.pro.licensechecker;

import java.util.EnumSet;
import java.util.Optional;

/**
 * A set of License Checker capabilities (license check option) that it can run
 * for a subscription and product. A capability is represented using the
 * {@link Capability} enum.
 *
 * @since 2.0.0
 */
public class Capabilities {
    private final EnumSet<Capability> capabilities;

    /**
     * Creates an empty set of capabilities.
     */
    public Capabilities() {
        this.capabilities = EnumSet.noneOf(Capability.class);
    }

    /**
     * Creates a set of capabilities with a single capability.
     *
     * @param capability
     *            the license check to add
     */
    public Capabilities(Capability capability) {
        this.capabilities = EnumSet.of(capability);
    }

    /**
     * Creates a set of capabilities with multiple capabilities.
     *
     * @param capabilities
     *            the capabilities to add
     */
    public Capabilities(Capability... capabilities) {
        this.capabilities = EnumSet
                .copyOf(EnumSet.of(capabilities[0], capabilities));
    }

    /**
     * Checks if this set of capabilities contains a given capability.
     *
     * @param capability
     *            the license checks to check
     * @return {@code true} if this set of capabilities contains the given
     *         capability, {@code false} otherwise
     */
    public boolean has(Capability capability) {
        return capabilities.contains(capability);
    }

    /**
     * Creates an empty set of capabilities.
     *
     * @return an empty set of capabilities
     */
    public static Capabilities none() {
        return new Capabilities();
    }

    /**
     * Creates a set of capabilities with a single capability.
     *
     * @param capability
     *            the license check to add
     * @return a set of capabilities with a single capability
     */
    public static Capabilities of(Capability capability) {
        return new Capabilities(capability);
    }

    /**
     * Creates a set of capabilities with multiple capabilities.
     *
     * @param capabilities
     *            the capabilities to add
     * @return a set of capabilities with multiple capabilities
     */
    public static Capabilities of(Capability... capabilities) {
        return new Capabilities(capabilities);
    }

    /**
     * Gets the optional pre-trial validator for this set of capabilities.
     *
     * @return the optional pre-trial validator for this set of capabilities or
     *         empty if pre-trial is not added to the capabilities.
     */
    Optional<PreTrialValidator> getPreTrialValidator() {
        return Optional.ofNullable(
                has(Capability.PRE_TRIAL) ? new PreTrialValidator() : null);
    }
}
