/*
 * Copyright 2010 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.pro.licensechecker.json.impl;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.List;

import com.vaadin.pro.licensechecker.json.LicenseCheckerJson;
import com.vaadin.pro.licensechecker.json.LicenseCheckerJsonArray;
import com.vaadin.pro.licensechecker.json.LicenseCheckerJsonBoolean;
import com.vaadin.pro.licensechecker.json.LicenseCheckerJsonFactory;
import com.vaadin.pro.licensechecker.json.LicenseCheckerJsonNumber;
import com.vaadin.pro.licensechecker.json.LicenseCheckerJsonObject;
import com.vaadin.pro.licensechecker.json.LicenseCheckerJsonString;
import com.vaadin.pro.licensechecker.json.LicenseCheckerJsonType;
import com.vaadin.pro.licensechecker.json.LicenseCheckerJsonValue;

/**
 * Server-side implementation of LicenseCheckerJsonArray.
 */
public class LicenseCheckerJreJsonArray extends LicenseCheckerJreJsonValue
        implements LicenseCheckerJsonArray {

    private static final long serialVersionUID = 1L;

    private transient ArrayList<LicenseCheckerJsonValue> arrayValues = new ArrayList<LicenseCheckerJsonValue>();

    private transient LicenseCheckerJsonFactory factory;

    public LicenseCheckerJreJsonArray(LicenseCheckerJsonFactory factory) {
        this.factory = factory;
    }

    @Override
    public boolean asBoolean() {
        return true;
    }

    @Override
    public double asNumber() {
        switch (length()) {
        case 0:
            return 0;
        case 1:
            return get(0).asNumber();
        default:
            return Double.NaN;
        }
    }

    @Override
    public String asString() {
        StringBuilder toReturn = new StringBuilder();
        for (int i = 0; i < length(); i++) {
            if (i > 0) {
                toReturn.append(", ");
            }
            toReturn.append(get(i).asString());
        }
        return toReturn.toString();
    }

    public LicenseCheckerJsonValue get(int index) {
        return arrayValues.get(index);
    }

    public LicenseCheckerJsonArray getArray(int index) {
        return (LicenseCheckerJsonArray) get(index);
    }

    public boolean getBoolean(int index) {
        return ((LicenseCheckerJsonBoolean) get(index)).getBoolean();
    }

    public double getNumber(int index) {
        return ((LicenseCheckerJsonNumber) get(index)).getNumber();
    }

    public LicenseCheckerJsonObject getObject(int index) {
        return (LicenseCheckerJsonObject) get(index);
    }

    public Object getObject() {
        List<Object> objs = new ArrayList<Object>();
        for (LicenseCheckerJsonValue val : arrayValues) {
            objs.add(((LicenseCheckerJreJsonValue) val).getObject());
        }
        return objs;
    }

    public String getString(int index) {
        return ((LicenseCheckerJsonString) get(index)).getString();
    }

    public LicenseCheckerJsonType getType() {
        return com.vaadin.pro.licensechecker.json.LicenseCheckerJsonType.ARRAY;
    }

    @Override
    public boolean jsEquals(LicenseCheckerJsonValue value) {
        return getObject()
                .equals(((LicenseCheckerJreJsonValue) value).getObject());
    }

    public int length() {
        return arrayValues.size();
    }

    @Override
    public void remove(int index) {
        arrayValues.remove(index);
    }

    public void set(int index, LicenseCheckerJsonValue value) {
        if (value == null) {
            value = factory.createNull();
        }
        if (index == arrayValues.size()) {
            arrayValues.add(index, value);
        } else {
            arrayValues.set(index, value);
        }
    }

    public void set(int index, String string) {
        set(index, factory.create(string));
    }

    public void set(int index, double number) {
        set(index, factory.create(number));
    }

    public void set(int index, boolean bool) {
        set(index, factory.create(bool));
    }

    public String toJson() {
        return LicenseCheckerJsonUtil.stringify(this);
    }

    @Override
    public void traverse(
            com.vaadin.pro.licensechecker.json.impl.LicenseCheckerJsonVisitor visitor,
            com.vaadin.pro.licensechecker.json.impl.LicenseCheckerJsonContext ctx) {
        if (visitor.visit(this, ctx)) {
            LicenseCheckerJsonArrayContext arrayCtx = new LicenseCheckerJsonArrayContext(
                    this);
            for (int i = 0; i < length(); i++) {
                arrayCtx.setCurrentIndex(i);
                if (visitor.visitIndex(arrayCtx.getCurrentIndex(), arrayCtx)) {
                    visitor.accept(get(i), arrayCtx);
                    arrayCtx.setFirst(false);
                }
            }
        }
        visitor.endVisit(this, ctx);
    }

    private void readObject(ObjectInputStream stream)
            throws IOException, ClassNotFoundException {
        LicenseCheckerJreJsonArray instance = parseJson(stream);
        this.factory = LicenseCheckerJson.instance();
        this.arrayValues = instance.arrayValues;
    }

    private void writeObject(ObjectOutputStream stream) throws IOException {
        stream.writeObject(toJson());
    }
}
