/*
 * Copyright 2000-2024 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.pro.licensechecker.dau;

/**
 * An enforcement rule dictated by the License Server.
 * <p>
 * </p>
 * If enforcement is active, DAU tracking should throw an
 * {@link EnforcementException} when the count of new users exceeds the limit
 * set by the License Server.
 */
final class EnforcementRule {

    private final boolean enforce;
    private final int rateLimitRemaining;

    public EnforcementRule(boolean enforce, int rateLimitRemaining) {
        this.enforce = enforce;
        this.rateLimitRemaining = rateLimitRemaining;
    }

    /**
     * Checks if enforcement should be applied, based on remaining rate and
     * current users.
     *
     * @param trackedUsers
     *            the number of tracked users
     * @return {@literal true} if enforcement is required and tracked users
     *         number exceeds the limit, {@literal false} otherwise.
     */
    public boolean shouldEnforce(int trackedUsers) {
        if (trackedUsers < 0) {
            throw new IllegalArgumentException(
                    "trackedUsers cannot be negative");
        }
        return enforce && trackedUsers > rateLimitRemaining;
    }

    /**
     * Returns the current enforcement status obtained from license server.
     *
     * @return {@literal true} if the license server responded with enforce=TRUE
     *         in the last round trip, {@literal false} otherwise.
     */
    public boolean isActiveEnforcement() {
        return enforce;
    }

    @Override
    public String toString() {
        return "Enforcement "
                + (enforce ? "ON (" + rateLimitRemaining + " remaining users)"
                        : "OFF");
    }

    @Override
    public boolean equals(Object o) {
        if (this == o)
            return true;
        if (o == null || getClass() != o.getClass())
            return false;

        EnforcementRule rule = (EnforcementRule) o;
        return enforce == rule.enforce
                && rateLimitRemaining == rule.rateLimitRemaining;
    }

    @Override
    public int hashCode() {
        int result = Boolean.hashCode(enforce);
        result = 31 * result + rateLimitRemaining;
        return result;
    }
}
