/*
 * Copyright 2000-2025 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.pro.licensechecker;

import java.util.Objects;

/**
 * Represents a pre-trial state for a trial subscription, containing information
 * about the trial name, state, remaining days, and days until renewal.
 */
public final class PreTrial {
    final String trialName;
    final PreTrialState trialState;
    final int daysRemaining;
    final int daysRemainingUntilRenewal;

    public PreTrial(String trialName, PreTrialState trialState,
            int daysRemaining, int daysRemainingUntilRenewal) {
        this.trialName = Objects.requireNonNull(trialName,
                "Trial name cannot be null");
        this.trialState = Objects.requireNonNull(trialState,
                "Trial state cannot be null");
        this.daysRemaining = daysRemaining;
        this.daysRemainingUntilRenewal = daysRemainingUntilRenewal;
    }

    /**
     * Retrieves the name of the trial associated with this pre-trial instance.
     *
     * @return the trial name as a String, never null
     */
    public String getTrialName() {
        return trialName;
    }

    /**
     * Retrieves the number of days remaining in the pre-trial period.
     *
     * @return the number of pre-trial remaining days
     */
    public int getDaysRemaining() {
        return daysRemaining;
    }

    /**
     * Retrieves the number of days remaining until a new pre-trial can be
     * started.
     *
     * @return the number of days remaining until pre-trial can be renewed.
     */
    public int getDaysRemainingUntilRenewal() {
        return daysRemainingUntilRenewal;
    }

    /**
     * Retrieves the current state of the pre-trial instance.
     *
     * @return the current {@link PreTrialState} of the pre-trial
     */
    public PreTrialState getTrialState() {
        return trialState;
    }

    /**
     * Represents the state of a pre-trial.
     */
    public enum PreTrialState {
        /**
         * Indicates that the pre-trial process is permitted to start.
         */
        START_ALLOWED,
        /**
         * Indicates that the pre-trial does not allow using a specific
         * commercial product.
         */
        ACCESS_DENIED,
        /**
         * Indicates that the pre-trial period is currently active and ongoing.
         */
        RUNNING,
        /**
         * Indicates that the pre-trial period has concluded and is no longer
         * active.
         */
        EXPIRED
    }

    @Override
    public String toString() {
        return "PreTrial {" + "trialName='" + trialName + '\'' + ", trialState="
                + trialState + ", daysRemaining=" + daysRemaining
                + ", daysRemainingUntilRenewal=" + daysRemainingUntilRenewal
                + '}';
    }
}
