/*
 * Copyright 2000-2024 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.pro.licensechecker;

import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.interfaces.ECPublicKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.X509EncodedKeySpec;
import java.text.ParseException;
import java.time.Instant;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;

import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.JWSVerifier;
import com.nimbusds.jose.crypto.ECDSAVerifier;
import com.nimbusds.jwt.JWTParser;
import com.nimbusds.jwt.SignedJWT;
import org.slf4j.Logger;

/**
 * Validator that can allow or deny usage of a given product version for a given
 * offline key.
 * <p>
 * For development, the offline key can be used if it has not expired and the
 * checksum matches the expected for the machine it is run on.
 * <p>
 * For production, the offline key can also be used if it is a productionOnly
 * key and has not expired.
 */
public class OfflineKeyValidator {

    private static final Map<String, String> PUBLIC_KEYS = new HashMap<>();
    static {
        // Staging
        PUBLIC_KEYS.put("b98c7421853a2d11fb2be2fb73f89be54414a4e9",
                "MIGbMBAGByqGSM49AgEGBSuBBAAjA4GGAAQB/qMpeWPlOKTd8+93GSpi3s/CQMx1"
                        + "gpkw728vl8iijo2965zIBD1bePNULT9VK1iul2iNJA2ev9ImXecLAA4UoMwAlz3t"
                        + "QHIA8zJksNbQUHZhzS74hH/jJr9pE6ra4Q3lnNvmJKEXkFvCpUoBmdYS94Hu0MFX"
                        + "Fi16IJfooLW6qzmtUGs=");
        // Production
        PUBLIC_KEYS.put("542764e7000908e65dc3fc1dabf4e2cd28966758",
                "MIGbMBAGByqGSM49AgEGBSuBBAAjA4GGAAQABaxDhdMljdpoM43y31co033oQjTZ"
                        + "oCj+Wjby9LRBPmdlvMTAJV6gXOzZHDpXQb4N1O0NJr4AeXxaE4GO/p4GGywAkg+S"
                        + "YIO1v8X+n2beq1czN+i8WL1cfUu8DFITUkSHtULtPyNTvW1Ew7XeTGVUQ6n/Xz2Y"
                        + "fAy7tcoFDsldrurE1nY=");
    }

    private static Logger getLogger() {
        return LicenseChecker.getLogger();
    }

    private static boolean isExpired(long expires) {
        return Instant.now().isAfter(Instant.ofEpochMilli(expires));
    }

    /**
     * Validates that the given offline license is valid and provides access to
     * the given product.
     *
     * @param product
     *            the product to validate
     * @param buildType
     *            the type of build: production or development
     * @param offlineKey
     *            the offline license key
     * @param machineId
     *            the machine ID where we are running
     *
     * @return {@code true} if the validation succeeded
     * @throws LicenseException
     *             if the validation fails because and invalid offline key was
     *             provided
     */
    boolean validate(Product product, BuildType buildType,
            OfflineKey offlineKey, MachineId machineId) {
        getLogger().debug("Offline validation using offlineKey for " + product);
        if (offlineKey == null) {
            getLogger().debug("No offline key found");
            return false;
        }

        if (product != null
                && History.isRecentlyValidated(product, buildType, null)) {
            // check only every 24h
            getLogger().debug(
                    "Skipping check as product license was recently validated.");
            return true;
        }

        validateOfflineKey(offlineKey, machineId);

        if (buildType != null) {
            // When buildType is null, usage should be allowed regardless of
            // production/development limitations in the license
            if (buildType == BuildType.DEVELOPMENT
                    && !offlineKey.isDevelopmentBuildAllowed()) {
                getLogger().debug("Offline key is not for development");
                throw new LicenseException(getNotDevelopmentMessage(machineId));

            }
            if (buildType == BuildType.PRODUCTION
                    && !offlineKey.isProductionBuildAllowed()) {
                getLogger().debug("Offline key is not for production builds");
                throw new LicenseException(
                        getNotProductionBuildMessage(machineId));
            }
        }

        String licenseMachineId = offlineKey.getMachineId();
        if (licenseMachineId != null) {
            // If there is a machine id, it must match. Otherwise it is a
            // special, broad offline license

            if (!machineId.stablePartMatches(licenseMachineId)) {
                getLogger().debug("Offline key has incorrect machine id");
                throw new LicenseException(
                        getInvalidOfflineKeyMessage(machineId));
            }
        }

        if (isExpired(offlineKey.getExpires())) {
            getLogger().debug("Offline key expired");
            throw new LicenseException(getExpiredOfflineKeyMessage(machineId));
        }

        if (product != null) {
            if (!containsProduct(offlineKey, product)) {
                throw new LicenseException(
                        "The offline key does not provide access to "
                                + product.getName() + " "
                                + product.getVersion());
            }
            History.setLastCheckTimeNow(product, buildType, null);
            History.setLastSubscription(product, offlineKey.getSubscription(),
                    null);
        }

        getLogger().debug("Offline key OK");
        return true;
    }

    private boolean containsProduct(OfflineKey offlineKey, Product product) {
        if (product.getName().startsWith("test-")) {
            // Allow all test products to pass
            return true;
        }
        boolean hasExtendedSupport = offlineKey.getAllowedFeatures()
                .contains("extendedsupport");
        if ("vaadin-framework".equals(product.getName())) {
            if (product.getVersion().startsWith("7")
                    && (hasExtendedSupport || offlineKey.getAllowedFeatures()
                            .contains("v7extendedsupport"))) {
                return true;
            }
            if (product.getVersion().startsWith("8")
                    && (hasExtendedSupport || offlineKey.getAllowedFeatures()
                            .contains("v8extendedsupport"))) {
                return true;
            }
        } else if ("flow".equals(product.getName())) {
            return hasExtendedSupport;
        }
        return offlineKey.getAllowedProducts().contains(product.getName());

    }

    void validateOfflineKey(OfflineKey offlineKey,
            MachineId machineIdForLinks) {
        try {
            String jwtData = offlineKey.getJwtData();

            // Claim kid = JWT.decode(jwtData).getHeaderClaim("kid");
            SignedJWT jwt = (SignedJWT) JWTParser.parse(jwtData);
            Object kid = jwt.getHeader().getKeyID();
            KeyFactory fact = KeyFactory.getInstance("EC");
            byte[] encoded = Base64.getDecoder().decode(PUBLIC_KEYS.get(kid));
            X509EncodedKeySpec keySpec = new X509EncodedKeySpec(encoded);
            ECPublicKey publicKey = (ECPublicKey) fact.generatePublic(keySpec);

            JWSVerifier verifier = new ECDSAVerifier(publicKey);
            if (!jwt.verify(verifier)) {
                getLogger().debug("Offline key failed verification");
                throw new LicenseException(
                        getInvalidOfflineKeyMessage(machineIdForLinks));
            }
        } catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
            getLogger().debug("Offline key could not be read", e);
            throw new LicenseException(
                    getErrorValidatingOfflineKeyMessage(machineIdForLinks), e);
        } catch (ParseException e) {
            getLogger().debug("Error parsing offline key", e);
            throw new LicenseException(
                    getInvalidOfflineKeyMessage(machineIdForLinks), e);
        } catch (JOSEException e) {
            getLogger().debug("Error reading offline key", e);
            throw new LicenseException(
                    getInvalidOfflineKeyMessage(machineIdForLinks), e);
        }

    }

    private static String getExpiredOfflineKeyMessage(MachineId machineId) {
        return "Offline key has expired, "
                + getOfflineKeyLinkMessage(machineId);
    }

    private static String getNotDevelopmentMessage(MachineId machineId) {
        return "The provided offline key does not allow development, "
                + getOfflineKeyLinkMessage(machineId);
    }

    private static String getNotProductionBuildMessage(MachineId machineId) {
        return "The provided offline key does not allow production builds, "
                + getOfflineKeyLinkMessage(machineId);
    }

    static String getOfflineKeyLinkMessage(MachineId machineId) {
        return "please go to " + getOfflineUrl(machineId)
                + " to retrieve an offline key.\n"
                + "For CI/CD build servers, you need to download a server license key, which can work offline to "
                + "create production builds. You can download a server license key from "
                + "https://vaadin.com/myaccount/licenses.\n"
                + "For troubleshooting steps, see https://vaadin.com/licensing-faq-and-troubleshooting.";
    }

    @Deprecated
    public static String getOfflineUrl(String machineId) {
        return "https://vaadin.com/pro/validate-license?getOfflineKey="
                + machineId;
    }

    public static String getOfflineUrl(MachineId machineId) {
        String url = "https://vaadin.com/pro/validate-license?getOfflineKey=";
        if (machineId != null) {
            url += machineId.getPrimaryIdFull();
        }

        return url;
    }

    private static String getErrorValidatingOfflineKeyMessage(
            MachineId machineId) {
        return "Unable to validate offline key, "
                + getOfflineKeyLinkMessage(machineId);
    }

    static String getInvalidOfflineKeyMessage(MachineId machineId) {
        return "Invalid offline key, " + getOfflineKeyLinkMessage(machineId);
    }

    static String getMissingOfflineKeyMessage(MachineId machineId) {
        return "The license server at " + Constants.LICENSE_SERVER_HOST
                + " could not be reached and no offline key was found. To use the product without an internet connection "
                + getOfflineKeyLinkMessage(machineId);
    }

}
