/*
 * Copyright 2000-2024 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.pro.licensechecker;

import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.nio.file.attribute.FileTime;
import java.util.Objects;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import oshi.SystemInfo;
import oshi.hardware.CentralProcessor;
import oshi.hardware.ComputerSystem;
import oshi.hardware.HardwareAbstractionLayer;

public class MachineId {

    private static final MachineIdFileAccess DEFAULT_FILE_ACCESS = new DefaultMachineIdFileAccess();

    // Instance fields
    private final String v1Id;
    private final String v2FullId;

    private MachineIdFileAccess fileAccess;

    /**
     * Generates both v1 and v2 machine IDs.
     */
    MachineId() {
        this(DEFAULT_FILE_ACCESS);
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(MachineId.class);
    }

    /**
     * Temporary, for migration. Only accepts v1 machine ID format.
     *
     * @param v1Id
     *            the v1 machine ID (must be v1 format, not v2)
     * @throws IllegalArgumentException
     *             if the provided ID is v2 format
     */
    @Deprecated
    MachineId(String v1Id) {
        this.fileAccess = DEFAULT_FILE_ACCESS;
        this.v1Id = Objects.requireNonNull(v1Id);
        if (isV2(v1Id)) {
            throw new IllegalArgumentException(
                    "String constructor only accepts v1 machine IDs. Got v2: "
                            + v1Id);
        }
        String v2 = "";
        try {
            v2 = generateV2();
        } catch (Throwable e) {
            getLogger().error("Failed to generate v2 Machine ID", e);
        }
        this.v2FullId = v2;
    }

    /**
     * For testing.
     *
     * @param v1Id
     *            the v1 machine ID
     * @param v2FullId
     *            the v2 full machine ID
     */
    MachineId(String v1Id, String v2FullId) {
        this.v1Id = Objects.requireNonNull(v1Id);
        this.v2FullId = Objects.requireNonNull(v2FullId);
    }

    /**
     * For testing with custom file access.
     *
     * @param fileAccess
     *            the file access implementation to use
     */
    MachineId(MachineIdFileAccess fileAccess) {
        this.fileAccess = fileAccess;
        String v1 = null;
        String v2 = "";
        try {
            v1 = generateV1();
        } catch (Throwable e) {
            getLogger().debug("Failed to generate v1 Machine ID", e);
        }
        try {
            v2 = generateV2();
        } catch (Throwable e) {
            getLogger().error("Failed to generate v2 Machine ID", e);
        }
        this.v1Id = v1;
        this.v2FullId = v2;

    }

    /**
     * Returns the v1 machine ID. Package-private - internal use only.
     *
     * @return the v1 machine ID
     */
    String getV1() {
        return v1Id;
    }

    /**
     * Returns the full v2 machine ID. Package-private - internal use only.
     *
     * @return the full v2 machine ID
     */
    String getV2Full() {
        return v2FullId;
    }

    /**
     * Returns the full machine id, which should be used for link generation and
     * online queries.
     *
     * @return the machine ID to use
     */
    String getPrimaryIdFull() {
        return getV2Full();
    }

    /**
     * Returns the stable part of the machine id, which should be used for pre
     * trials.
     *
     * @return the machine ID to use
     */
    String getPrimaryIdStable() {
        return extractStablePart(getV2Full());
    }

    /**
     * Checks if this machine ID matches the provided license machine ID.
     * Handles both v1 and v2 license IDs automatically. Package-private -
     * internal use only.
     *
     * @param licenseMachineId
     *            the machine ID from the license
     * @return true if this machine ID matches the license machine ID
     * @throws IllegalStateException
     *             if license requires v1 ID but OSHI is not available
     */
    boolean stablePartMatches(String licenseMachineId) {
        if (licenseMachineId == null || licenseMachineId.isEmpty()) {
            return false;
        }

        if (isV2(licenseMachineId)) {
            // V2 license: compare stable parts OR check if it matches v1 ID
            // (for cross-version compatibility)
            return equalsV2StableId(licenseMachineId);
        } else if (isV1(licenseMachineId)) {
            // V1 license: requires v1 ID to be available
            if (v1Id == null) {
                String message = "Cannot generate V1 Machine ID: OSHI library not found.\n"
                        + "OSHI is required for V1 Machine IDs but will be made optional in a future version.\n"
                        + "Either:\n"
                        + "1. Add OSHI dependency to your project:\n"
                        + "   <dependency>\n"
                        + "     <groupId>com.github.oshi</groupId>\n"
                        + "     <artifactId>oshi-core</artifactId>\n"
                        + "     <version>6.9.1</version>\n"
                        + "   </dependency>\n"
                        + "2. Generate a new V2 offline license at: "
                        + OfflineKeyValidator.getOfflineUrl(v2FullId);
                throw new IllegalStateException(message);
            }
            // V1 license: compare full IDs
            return licenseMachineId.equals(v1Id);
        }

        return false;
    }

    /**
     * Compares the stable part of a v2 machine ID with this machine's v2 stable
     * ID. Package-private - internal use only.
     *
     * @param machineId
     *            the machine ID to compare
     * @return true if the stable parts match
     */
    boolean equalsV2StableId(String machineId) {
        if (machineId == null || machineId.isEmpty()) {
            return false;
        }
        String currentStable = extractStablePart(v2FullId);
        String providedStable = extractStablePart(machineId);
        return currentStable.equals(providedStable);
    }

    /**
     * Compares the full v2 machine ID with this machine's v2 ID.
     * Package-private - internal use only.
     *
     * @param machineId
     *            the machine ID to compare
     * @return true if the full v2 IDs match
     */
    boolean equalsV2FullId(String machineId) {
        if (machineId == null || machineId.isEmpty()) {
            return false;
        }
        return v2FullId.equals(machineId);
    }

    /**
     * Compares the v1 machine ID with this machine's v1 ID. Package-private -
     * internal use only.
     *
     * @param machineId
     *            the machine ID to compare
     * @return true if the v1 IDs match
     */
    boolean equalsV1(String machineId) {
        if (machineId == null || machineId.isEmpty()) {
            return false;
        }
        return v1Id.equals(machineId);
    }

    public static void main(String[] args) {
        // Default behavior: print v1 only (backwards compatible)
        if (args.length == 0) {
            System.out.println(new MachineId().getV1());
            return;
        }

        // --v2 flag: show v2 information
        if (args.length > 0 && "--v2".equals(args[0])) {
            MachineId machineId = new MachineId();
            System.out.println("Machine ID v1 (current default):");
            System.out.println("  " + machineId.getV1());
            System.out.println();
            System.out.println("Machine ID v2:");
            System.out.println("  Full ID:   " + machineId.getV2Full());
            System.out.println(
                    "  Stable ID: " + extractStablePart(machineId.getV2Full()));
        } else {
            System.out.println("Usage: java MachineId [--v2]");
            System.out.println("  (no args): Print v1 Machine ID");
            System.out.println(
                    "  --v2:      Print detailed v1 and v2 information");
        }
    }

    // Public static API

    /**
     * Returns the primary machine ID as a string.
     *
     * @return the machine ID
     * @deprecated Use {@link #MachineId()} and instance methods instead.
     */
    @Deprecated
    public static String get() {
        return new MachineId().getPrimaryIdFull();
    }

    // Private generation methods

    private static String generateV1() {
        try {
            SystemInfo systemInfo = new SystemInfo();
            HardwareAbstractionLayer hardwareAbstractionLayer = systemInfo
                    .getHardware();

            ComputerSystem computerSystem = hardwareAbstractionLayer
                    .getComputerSystem();
            String uuid = computerSystem.getHardwareUUID();

            CentralProcessor centralProcessor = hardwareAbstractionLayer
                    .getProcessor();
            String processorIdentifier = centralProcessor
                    .getProcessorIdentifier().getIdentifier();

            String delimiter = "-";

            return "mid-" + String.format("%08x", uuid.hashCode()) + delimiter
                    + String.format("%08x", processorIdentifier.hashCode());
        } catch (NoClassDefFoundError e) {
            // OSHI library not available - v1 Machine ID cannot be generated
            getLogger().debug(
                    "OSHI library not found, V1 Machine ID generation not available",
                    e);
            return null;
        }
    }

    /**
     * Generates a V2 Machine ID using the provided file access implementation.
     * Package-private for testing.
     *
     * @return the generated Machine ID
     * @throws Exception
     *             if an error occurs during generation
     */
    String generateV2() throws Exception {
        Path homePath = fileAccess.getUserHome();
        String homePathStr = homePath.toAbsolutePath().toString();

        MachineIdFileAccess.FileAttributes attrs = fileAccess
                .getFileAttributes(homePath);
        FileTime creationTime = attrs.getCreationTime();
        FileTime modifiedTime = attrs.getLastModifiedTime();

        long timestamp = creationTime.toMillis();
        String stablePart;
        String uniquePart;

        // If creation time differs from modification time, filesystem supports
        // creation time - include it in stable part and use zeros for unique
        // part
        if (!creationTime.equals(modifiedTime)) {
            // Creation time is supported: include it in stable part
            stablePart = Util.getHashPrefix(homePathStr + ":" + timestamp, null,
                    StandardCharsets.UTF_8, 8);
            // Unstable part is zeros when creation time is in stable part
            uniquePart = "00000000";
        } else {
            // Creation time not supported: use only home path in stable part,
            // and put timestamp in unique part
            stablePart = Util.getHashPrefix(homePathStr, null,
                    StandardCharsets.UTF_8, 8);
            uniquePart = Util.getHashPrefix(String.valueOf(timestamp), null,
                    StandardCharsets.UTF_8, 8);
        }

        return "mid2-" + stablePart + "-" + uniquePart;
    }

    // Static utility methods

    static boolean isV1(String machineId) {
        if (machineId == null) {
            return false;
        }
        // Special case for legacy test machine ID "aa1122" used in test JWTs
        // TODO: Remove this when test JWTs are regenerated with proper format
        if ("aa1122".equals(machineId)) {
            return true;
        }
        return machineId.startsWith("mid-");
    }

    static boolean isV2(String machineId) {
        return machineId != null && machineId.startsWith("mid2-");
    }

    static String extractStablePart(String machineId) {
        if (machineId == null) {
            return "";
        }
        if (isV1(machineId)) {
            // V1: return full ID (no stable part concept)
            return machineId;
        } else if (isV2(machineId)) {
            // V2: extract "mid2-[stablePart]"
            // Format can be "mid2-xxxxxxxx-yyyyyyyy" or "mid2-xxxxxxxx"
            // Count dashes to determine if it already is stable part only
            int firstDash = machineId.indexOf('-');
            int lastDash = machineId.lastIndexOf('-');

            if (firstDash == lastDash) {
                // Only one dash: already stable part (mid2-xxxxxxxx)
                return machineId;
            } else {
                // Two dashes: full ID (mid2-xxxxxxxx-yyyyyyyy)
                return machineId.substring(0, lastDash);
            }
        }
        return machineId;
    }

}
