/*
 * Copyright 2010 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.pro.licensechecker.json.impl;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;

import com.vaadin.pro.licensechecker.json.LicenseCheckerJsonString;
import com.vaadin.pro.licensechecker.json.LicenseCheckerJsonType;
import com.vaadin.pro.licensechecker.json.LicenseCheckerJsonValue;

/**
 * Server-side implementation of LicenseCheckerJsonString.
 */
public class LicenseCheckerJreJsonString extends LicenseCheckerJreJsonValue
        implements LicenseCheckerJsonString {

    private static final long serialVersionUID = 1L;

    private transient String string;

    public LicenseCheckerJreJsonString(String string) {
        this.string = string;
    }

    @Override
    public boolean asBoolean() {
        return !getString().isEmpty();
    }

    @Override
    public double asNumber() {
        try {
            if (asString().isEmpty()) {
                return 0.0;
            } else {
                return Double.parseDouble(asString());
            }
        } catch (NumberFormatException nfe) {
            return Double.NaN;
        }
    }

    @Override
    public String asString() {
        return getString();
    }

    public Object getObject() {
        return getString();
    }

    public String getString() {
        return string;
    }

    public LicenseCheckerJsonType getType() {
        return LicenseCheckerJsonType.STRING;
    }

    @Override
    public boolean jsEquals(LicenseCheckerJsonValue value) {
        return getObject()
                .equals(((LicenseCheckerJreJsonValue) value).getObject());
    }

    @Override
    public void traverse(LicenseCheckerJsonVisitor visitor,
            LicenseCheckerJsonContext ctx) {
        visitor.visit(getString(), ctx);
    }

    public String toJson() throws IllegalStateException {
        return LicenseCheckerJsonUtil.quote(getString());
    }

    private void readObject(ObjectInputStream stream)
            throws IOException, ClassNotFoundException {
        LicenseCheckerJreJsonString instance = parseJson(stream);
        this.string = instance.string;
    }

    private void writeObject(ObjectOutputStream stream) throws IOException {
        stream.writeObject(toJson());
    }
}
